#ifndef NR_MouseInput_H
#define NR_MouseInput_H
#pragma once

#include <array>

#include "NR_InputObject.h"


namespace netrush	//NetRush context
{
	/// Button ID for mouse devices (from OIS::MouseButtonID, same values)
	enum MouseButton 
	{
		MOUSE_BUTTON_LEFT = 0,
		MOUSE_BUTTON_RIGHT,
		MOUSE_BUTTON_MIDDLE,
		MOUSE_BUTTON_3,
		MOUSE_BUTTON_4,
		MOUSE_BUTTON_5,
		MOUSE_BUTTON_6,
		MOUSE_BUTTON_7
	};

	/// Mouse position.
	union MousePos 
	{
		struct 
		{
			int x;
			int y;
			int z;
		};
		std::tr1::array<int, 3> pos;
	};

	/** Mouse inputs interface.
	*/
	class NETRUSH_SYSTEM_API MouseInput : public InputObject
	{
	public :
		///Maximum number of buttons.
		static const unsigned short MAX_MOUSE_BUTTONS = 8;

	private:	
		
	
	protected:

		/// Position in the screen.
		MousePos m_ScreenPosition;

		/// Movement since the last update.
		MousePos m_Move;

		/// Mouse buttons state.
		std::tr1::array<unsigned short, MAX_MOUSE_BUTTONS> m_ButtonState;
	
	
	public:

		/** Mouse buttons state.
			@param button Button to get the state from.
			@return State of the button.
		*/
		KeyState getButtonState(MouseButton button) const
		{
			return static_cast<KeyState>(m_ButtonState[button]);
		}

		/** Mouse buttons state.
			@param button Button that must be modified.
			@param state New state of the button.
		*/
		void setButtonState(MouseButton button, KeyState state)
		{
			m_ButtonState[button]=state;
		}

		// helper functions
		bool isButtonDown( MouseButton button ) const { return getButtonState( button ) == KEY_IS_DOWN; }
		bool isButtonUp( MouseButton button ) const { return getButtonState( button ) == KEY_IS_UP; }
		bool isButtonHold( MouseButton button ) const { return getButtonState( button ) == KEY_IS_HOLD; }
		bool isButtonUsed( MouseButton button ) const { return getButtonState( button ) != KEY_IS_NOT_USED; }


		/** Reset the buttons to not used. */
		void resetButtonState()
		{
			m_ButtonState.assign( KEY_IS_NOT_USED );
		}

		/// Position of the mouse on the screen (pixels).
		const MousePos& getScreenPosition() const
		{ return m_ScreenPosition; }

		/// Movement since the last update.
		const MousePos& getMove() const
		{ return m_Move;}
		
		/** Position on the screen (pixels).
			@param New position of the mouse.
		*/
		void setScreenPosition(const MousePos& pos)
		{ 
			m_ScreenPosition.x=pos.x;
			m_ScreenPosition.y=pos.y;
			m_ScreenPosition.z=pos.z;	 
		}

		/** Reset the screen position. */
		void resetScreenPosition()
		{
			m_ScreenPosition.x=0;
			m_ScreenPosition.y=0;
			m_ScreenPosition.z=0;
		}

		/** Movement since the last update.
			@param domove Move to apply to the mouse.
		*/
		void setMove(const MousePos& domove)
		{ 
			m_Move.x=domove.x;
			m_Move.y=domove.y;
			m_Move.z=domove.z;		
		}

		/** Reset movement since the last update. */
		void resetMove()
		{
			m_Move.x=0;
			m_Move.y=0;
			m_Move.z=0;
		}

		/// Reset data of the input
		void resetState()
		{
			resetScreenPosition();
			resetMove();
			resetButtonState();
		}

		/** Set the screen size.
		*/
		virtual void setScreenSize(int width, int height)=0;

		/** Constructor.
		*/
		MouseInput() { resetState(); }
		
		/** Destruction.
		*/
		virtual ~MouseInput(){};	
	
	};

}

#endif	//NR_MouseInput_H