#include <sstream>

#include "NR_InputManager.h"

#include "GC_TaskManager.h"

//using OIS :
#include "OISInputManager.h"
#include "OISException.h"

#include "NR_System.h"
#include "NR_InputManager.h"
#include "NR_Keyboard_OIS.h"
#include "NR_Mouse_OIS.h"


namespace netrush	//NetRush context
{

	InputManager::InputManager():
		m_Ready(false),
		m_WindowHandle(0),
		m_inputManager(0)
	{
		//Using OIS :
		m_Keyboard = new Keyboard_OIS();
		m_Mouse = new Mouse_OIS();
		
		m_taskInputUpdate = new Task_InputUpdate();
		
		NR_LOG << "InputManager created.";

		
	}

	
	InputManager::~InputManager()//InputManager destructor
	{
		if(m_Ready)
		{
			//Release input objects :
			stop();
		}

		NR_ASSERT( m_taskInputUpdate, "Tried to destroy an uninitialized input manager!" );

		NR_ASSERT( m_Mouse, "Tried to destroy an uninitialized input manager!");
		NR_ASSERT( m_Keyboard, "Tried to destroy an uninitialized input manager!");

		delete m_taskInputUpdate;

		delete m_Mouse;
		delete m_Keyboard;

		NR_LOG << "InputManager destroyed.";

	}

	/*
		Initialize inputs.
		Return false if one or more inputs initialization failed.
	*/
	void InputManager::initializeInputs(gcore::TaskManager& taskManager)
	{
		//////////////////////////////////////////////////////////////////////////
		//Initialize Input management:

		NR_LOG << "InputManager initialization...";


		OIS::ParamList paramlist;
		
		{
			std::stringstream wnd;
			wnd << static_cast<size_t>( m_WindowHandle ); 
			paramlist.insert(std::make_pair( std::string("WINDOW"), wnd.str() ));
		}
		
		
#if GC_PLATFORM == GC_PLATFORM_WIN32
		paramlist.insert(std::make_pair(std::string("w32_mouse"), std::string("DISCL_FOREGROUND" )));
		paramlist.insert(std::make_pair(std::string("w32_mouse"), std::string("DISCL_EXCLUSIVE")));
		paramlist.insert(std::make_pair(std::string("w32_keyboard"), std::string("DISCL_FOREGROUND")));
		paramlist.insert(std::make_pair(std::string("w32_keyboard"), std::string("DISCL_NONEXCLUSIVE")));
		paramlist.insert(std::make_pair(std::string("w32_keyboard"), std::string("DISCL_NOWINKEY")));
#elif GC_PLATFORM == GC_PLATFORM_LINUX
		paramlist.insert(std::make_pair(std::string("x11_mouse_grab"), std::string("false")));
		paramlist.insert(std::make_pair(std::string("x11_mouse_hide"), std::string("false")));
		paramlist.insert(std::make_pair(std::string("x11_keyboard_grab"), std::string("false")));
		paramlist.insert(std::make_pair(std::string("XAutoRepeatOn"), std::string("true")));
#elif
		NR_ASSERT( false, "No input config for Apple OSX!" );
#endif

		try
		{
			m_inputManager = OIS::InputManager::createInputSystem(paramlist);
			NR_ASSERT( m_inputManager != nullptr , "Input manager creation failed!" );
		}
		catch (OIS::Exception& ex)
		{
			//OIS::InputManager creation failed!!!
			NR_EXCEPTION << "OIS::InputManager creation failed!\n OIS Error : " << ex.what();
		}
		
		//////////////////////////////////////////////////////////////////////////
		//Initialize inputs:
		
		//Keyboard :
		m_Keyboard->initializeInput();
		NR_LOG << "Keyboard initialized.";
		
		//Mouse:
		m_Mouse->initializeInput();
		NR_LOG << "Mouse initialized.";

		NR_LOG << "InputManager initialized.";

		// register tasks
		taskManager.registerTask( *m_taskInputUpdate );	
		
	}

	/*
		Release inputs.
		
	*/
	void InputManager::releaseInputs()
	{
		//////////////////////////////////////////////////////////////////////////
		//Release inputs:
		
		//Mouse :
		if(m_Mouse->isReady() ) m_Mouse->releaseInput();
		NR_LOG << "Mouse released.";

		//Keyboard :
		if(m_Keyboard->isReady()) m_Keyboard->releaseInput();
		NR_LOG << "Keyboard released.";
		
		//task
		m_taskInputUpdate->unregister();

		//////////////////////////////////////////////////////////////////////////
		//Release Input management:
		NR_ASSERT( m_inputManager != nullptr , "Tried to release an uninitialized input manager!" );
		OIS::InputManager::destroyInputSystem(m_inputManager);
		m_inputManager = nullptr;


		NR_LOG << "InputManager released.";
	}

	/*
		Update inputs state.
	*/
	void InputManager::updateInputs()
	{
		NR_ASSERT( m_Mouse, "Tried to update uninitilized inputs!" );
		NR_ASSERT( m_Keyboard, "Tried to update uninitilized inputs!" );
		//Keyboard:
		if(m_Keyboard->isReady())m_Keyboard->updateInput();
		
		//Mouse:
		if(m_Mouse->isReady())m_Mouse->updateInput();
	}

	/*
		start acquiring inputs state and sending events using mapping definitions.
		This will initialize inputs and launch input management tasks.
	*/
	void InputManager::start(WindowHandle wHandle, gcore::TaskManager& taskManager )
	{
		
		//InputManager already started!!
		NR_ASSERT( !m_Ready , "InputManager already started!" );
		
		//We must have a valid window handle!
		NR_ASSERT( wHandle , "Invalid WindowHandle!" );
		m_WindowHandle=wHandle;

		//Initialization...
		initializeInputs( taskManager );
		
		//Launch tasks :
		//Input Update task
		m_taskInputUpdate->activate();

		m_Ready = true;
		NR_LOG << "Input management started.";

	}

	/*
		Stop acquiring inputs state and sending events using mapping definitions.
		This will kill input management tasks and release inputs.
	*/
	void InputManager::stop()
	{
		NR_ASSERT(m_Ready, "Tried to stop an uninitialized input manager!" );

		NR_ASSERT( m_taskInputUpdate, "Tried to stop an uninitialized input manager!" );

		NR_ASSERT( m_Mouse, "Tried to stop an uninitialized input manager!" );
		NR_ASSERT( m_Keyboard, "Tried to stop an uninitialized input manager!" );

		//Kill tasks :
		//Input Update task
		m_taskInputUpdate->terminate();

		m_Ready=false;
		m_WindowHandle=0;

		//Release inputs :
		releaseInputs();


		NR_LOG << "Input management stoped.";
		
	}

	/*
		Start acquiring inputs state and sending events using mapping definitions.
		This will just pause input management tasks.
	*/
	void InputManager::pause()
	{
		m_taskInputUpdate->pause();
		
		// invalidate any input state
		m_Keyboard->resetState();
		m_Mouse->resetState();

		NR_LOG << "Input management paused.";
	}

	/*
		Stop acquiring inputs state and sending events using mapping definitions.
		This will just resume input management tasks.
	*/
	void InputManager::resume()
	{
		m_taskInputUpdate->resume();
		NR_LOG << "Input management resumed.";

	}

	bool InputManager::isActive() const
	{
		NR_ASSERT_NOT_NULL( m_taskInputUpdate );
		return isReady() && m_taskInputUpdate->isActive();
	}
}// namespace netrush