<?php

class ReflectionView implements Reflector {
   
    protected $_pdo;
    
    protected $_meta_inf;
    
    public function __construct ($viewname, PDO $pdo) {
        if ($pdo === null)
            throw new InvalidArgumentException("Second argument must be a valid PDO instance");
        
        $this->_pdo = $pdo;
        
        if (is_string($viewname)) {
            if (strpos($viewname, '`') !== false) $viewname = str_replace('`', '', $viewname);
            list($schema,$view) = explode('.', $viewname);
        }
        elseif (is_array($viewname)) {
            list($schema,$view) = $viewname;
        }
        else {
            throw new InvalidArgumentException("First argument must be string or array, " . gettype($viewname) . " given");
        }
        
        $query = "SELECT `TABLE_CATALOG`,`TABLE_SCHEMA`,`TABLE_NAME`,`VIEW_DEFINITION`,`CHECK_OPTION`,`IS_UPDATABLE`,`DEFINER`,".
        		 "`SECURITY_TYPE`,`CHARACTER_SET_CLIENT`,`COLLATION_CONNECTION` FROM `information_schema`.`VIEWS` WHERE ".
        		 "`TABLE_SCHEMA`=:schema AND`TABLE_NAME`=:view";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $schema, PDO::PARAM_STR);
        $stmt->bindParam(':view',   $view,   PDO::PARAM_STR);
        if ($stmt->execute()) {
            if (!$stmt->rowCount())
                throw new ReflectionException("View `$schema`.`$view` does not exist");
            
            $this->_meta_inf = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    public function getCatalog () {
        return $this->_meta_inf['TABLE_CATALOG'];
    }
    
    public function getBase () {
        return new ReflectionBase ($this->_meta_inf['TABLE_SCHEMA'], $this->_pdo);
    }
    
    public function getName () {
        return $this->_meta_inf['TABLE_NAME'];
    }
    
    public function getDefinition () {
        return $this->_meta_inf['VIEW_DEFINITION'];
    }
    
    public function getCheckOption () {
        return $this->_meta_inf['CHECK_OPTION'];
    }
    
    public function isUpdatable () {
        return $this->_meta_inf['IS_UPDATABLE'] == 'YES';
    }
    
    public function getDefiner () {
        return $this->_meta_inf['DEFINER'];
    }
    public function getSecurityType () {
        return $this->_meta_inf['SECURITY_TYPE'];
    }
    
    public function getCharsetClient () {
        return $this->_meta_inf['CHARACTER_SET_CLIENT'];
    }
    
    public function getCollationConnection () {
        return $this->_meta_inf['COLLATION_CONNECTION'];
    }
    
    /**
     * -------------------------------------------------------------------------------------------------------------------------------------
     * Reflector methods
     */
    
    public static function export ($viewname, PDO $pdo, $return = false) {
        $view = new self ($viewname, $pdo);
        if ($return) {
            return (string)$view;
        }
        else {
            echo $view;
        }
    }
     
    public function __toString () {
        return "View [  view {$this->getName()} ]";
    }
    
}