<?php

class ReflectionTable implements Reflector {
    
    protected $_pdo;
    
    protected $_meta_inf;
    
    public function __construct ($tablename, PDO $pdo) {
        if ($pdo === null)
            throw new InvalidArgumentException("Second argument must be a valid PDO instance");
        
        $this->_pdo = $pdo;
        
        if (is_string($tablename)) {
            if (strpos($tablename, '`') !== false) $tablename = str_replace('`', '', $tablename);
            list($schema,$table) = explode('.', $tablename);
        }
        elseif (is_array($tablename)) {
            list($schema,$table) = $tablename;
        }
        else {
            throw new InvalidArgumentException("First argument must be string or array, " . gettype($tablename) . " given");
        }
        
        $query = "SELECT `TABLE_CATALOG`,`TABLE_SCHEMA`,`TABLE_NAME`,`TABLE_TYPE`,`ENGINE`,`VERSION`,`ROW_FORMAT`,`TABLE_ROWS`,".
        		 "`AVG_ROW_LENGTH`,`DATA_LENGTH`,`MAX_DATA_LENGTH`,`INDEX_LENGTH`,`DATA_FREE`,`AUTO_INCREMENT`,`CREATE_TIME`,".
        		 "`UPDATE_TIME`,`CHECK_TIME`,`TABLE_COLLATION`,`CHECKSUM`,`CREATE_OPTIONS`,`TABLE_COMMENT`".
                 " FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA`=:schema AND `TABLE_NAME`=:table";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $schema, PDO::PARAM_STR);
        $stmt->bindParam(':table', $table, PDO::PARAM_STR);
        if ($stmt->execute()) {
            if (!$stmt->rowCount())
                throw new ReflectionException("Table `$schema`.`$table` does not exist");
            
            $this->_meta_inf = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }

    public function getCatalog () {
        return $this->_meta_inf['TABLE_CATALOG'];
    }
    
    public function getName () {
        return $this->_meta_inf['TABLE_NAME'];
    }
    
    public function getType () {
        return $this->_meta_inf['TABLE_TYPE'];
    }
    
    public function getEngine () {
        return $this->_meta_inf['ENGINE'];
    }
    
    public function getVersion () {
        return $this->_meta_inf['VERSION'];
    }
    
    public function getRowFormat () {
        return $this->_meta_inf['ROW_FORMAT'];
    }
    
    public function getAverageRowLength () {
        return $this->_meta_inf['AVG_ROW_LENGTH'];
    }
    
    public function getDataLength () {
        return $this->_meta_inf['DATA_LENGTH'];
    }
    
    public function getMaxDataLength () {
        return $this->_meta_inf['MAX_DATA_LENGTH'];
    }
    
    public function getIndexLength () {
        return $this->_meta_inf['INDEX_LENGTH'];
    }
    
    public function getDataFree () {
        return $this->_meta_inf['DATA_FREE'];
    }
    
    public function getAutoIncrement () {
        return $this->_meta_inf['AUTO_INCREMENT'];
    }
    
    public function getCreateTime () {
        return $this->_meta_inf['CREATE_TIME'];
    }
    
    public function getUpdateTime () {
        return $this->_meta_inf['UPDATE_TIME'];
    }
    
    public function getCheckTime () {
        return $this->_meta_inf['CHECK_TIME'];
    }
    
    public function getCollation () {
        return $this->_meta_inf['TABLE_COLLATION'];
    }
    
    public function getChecksum () {
        return $this->_meta_inf['CHECKSUM'];
    }
    
    public function getCreateOptions () {
        return $this->_meta_inf['CREATE_OPTIONS'];
    }
    
    public function getComment () {
        return $this->_meta_inf['TABLE_COMMENT'];
    }

    public function getBase () {
        return new ReflectionBase($this->_meta_inf['TABLE_SCHEMA'], $this->_pdo);
    }
    
    public function getColumn ($rowname) {
        return new ReflectionColumn(array($this->_meta_inf['TABLE_SCHEMA'], $this->_meta_inf['TABLE_NAME'], $rowname), $this->_pdo);
    }
    
    public function getColumns () {
        $query = "SELECT `TABLE_SCHEMA`,`TABLE_NAME`,`COLUMN_NAME` FROM `information_schema`.`COLUMNS`".
                 " WHERE `TABLE_SCHEMA`=:schema AND `TABLE_NAME`=:table";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $this->_meta_inf['TABLE_SCHEMA'], PDO::PARAM_STR);
        $stmt->bindParam(':table',  $this->_meta_inf['TABLE_NAME'],   PDO::PARAM_STR);
        if ($stmt->execute()) {
            $list = array();
            foreach ($stmt as $row) {
                $list[] = new ReflectionColumn($row, $this->_pdo);
            }
            return $list;
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    public function getCreateTable () {
        $query = "SHOW CREATE TABLE `{$this->_meta_inf['TABLE_SCHEMA']}`.`{$this->_meta_inf['TABLE_NAME']}`";
        
        if ($stmt = $this->_pdo->query($query)) {
            if (!$stmt->rowCount())
                throw new ReflectionException("Cannot retrieve the SQL create statement");
                
            list($table,$create_table) = $stmt->fetch(PDO::FETCH_NUM);
            return $create_table;
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("Query execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    public function check () {
        $query = "CHECK TABLE `{$this->_meta_inf['TABLE_SCHEMA']}`.`{$this->_meta_inf['TABLE_NAME']}`";
        
        if ($stmt = $this->_pdo->query($query)) {
            if (!$stmt->rowCount())
                throw new ReflectionException("Cannot execute SQL check operation");
                
            list($table,$op,$msg_type,$msg_text) = $stmt->fetch(PDO::FETCH_ASSOC);
            return strtolower($msg_text) == 'ok';
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("Query execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
	/**
     * -------------------------------------------------------------------------------------------------------------------------------------
     * Reflector methods
     */
    
    public static function export ($tablename, PDO $pdo, $return = false) {
        $table = new self ($tablename, $pdo);
        if ($return) {
            return (string)$table;
        }
        else {
            echo $table;
        }
    }
     
    public function __toString () {
        $columns = $this->getColumns();
        $count = count($columns);
        return "Table [  table {$this->getName()} ] { Columns [$count] { ". implode(' ', $columns) ." } }";
    }
}