<?php

class ReflectionRoutine implements Reflector {
    
    protected $_pdo;
    
    protected $_meta_inf;
    
    public function __construct ($routinename, PDO $pdo) {
        if ($pdo === null)
            throw new InvalidArgumentException("Second argument must be a valid PDO instance");
        
        $this->_pdo = $pdo;
        
        if (is_string($routinename)) {
            if (strpos($routinename, '`') !== false) $routinename = str_replace('`', '', $routinename);
            list($schema,$routine) = explode('.', $routinename);
        }
        elseif (is_array($viewname)) {
            list($schema,$routine) = $routinename;
        }
        else {
            throw new InvalidArgumentException("First argument must be string or array, " . gettype($routinename) . " given");
        }
        
        $query = "SELECT `SPECIFIC_NAME`,`ROUTINE_CATALOG`,`ROUTINE_SCHEMA`,`ROUTINE_NAME`,`ROUTINE_TYPE`,`DTD_IDENTIFIER`,`ROUTINE_BODY`,".
        		 "`ROUTINE_DEFINITION`,`EXTERNAL_NAME`,`EXTERNAL_LANGUAGE`,`PARAMETER_STYLE`,`IS_DETERMINISTIC`,`SQL_DATA_ACCESS`,".
        		 "`SQL_PATH`,`SECURITY_TYPE`,`CREATED`,`LAST_ALTERED`,`SQL_MODE`,`ROUTINE_COMMENT`,`DEFINER`,`CHARACTER_SET_CLIENT`,".
        		 "`COLLATION_CONNECTION`,`DATABASE_COLLATION` FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_SCHEMA`=:schema AND ".
                 "`ROUTINE_NAME`=:routine";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema',  $schema,  PDO::PARAM_STR);
        $stmt->bindParam(':routine', $routine, PDO::PARAM_STR);
        if ($stmt->execute()) {
            if (!$stmt->rowCount())
                throw new ReflectionException("View `$schema`.`$view` does not exist");
            
            $this->_meta_inf = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    public function getSpecificName () {
        return $this->_meta_inf['SPECIFIC_NAME'];
    }
    
    public function getCatalog () {
        return $this->_meta_inf['ROUTINE_CATALOG'];
    }
    
    public function getBase () {
        return new ReflectionBase($this->_meta_inf['ROUTINE_SCHEMA'], $this->_pdo);
    }
    
    public function getName () {
        return $this->_meta_inf['ROUTINE_NAME'];
    }
    
    public function getType () {
        return $this->_meta_inf['ROUTINE_TYPE'];
    }
    
    public function getDTDIdentifier () {
        return $this->_meta_inf['DTD_IDENTIFIER'];
    }
    
    public function getBody () {
        return $this->_meta_inf['ROUTINE_BODY'];
    }
    
    public function getDefinition () {
        return $this->_meta_inf['ROUTINE_DEFINITION'];
    }
    
    public function getExternalName () {
        return $this->_meta_inf['EXTERNAL_NAME'];
    }
    
    public function getExternalLanguage () {
        return $this->_meta_inf['EXTERNAL_LANGUAGE'];
    }
    
    public function getParameterStyle () {
        return $this->_meta_inf['PARAMETER_STYLE'];
    }
    
    public function isDeterministic () {
        return $this->_meta_inf['IS_DETERMINISTIC'];
    }
    
    public function getSQLDataAccess () {
        return $this->_meta_inf['SQL_DATA_ACCESS'];
    }
    
    public function getSQLPath () {
        return $this->_meta_inf['SQL_PATH'];
    }
    
    public function getSecurityType () {
        return $this->_meta_inf['SECURITY_TYPE'];
    }
    
    public function getCreateTime () {
        return $this->_meta_inf['CREATED'];
    }
    
    public function getUpdateTime () {
        return $this->_meta_inf['LAST_ALTERED'];
    }
    
    public function getSQLMode () {
        return $this->_meta_inf['SQL_MODE'];
    }
    
    public function getComment () {
        return $this->_meta_inf['ROUTINE_COMMENT'];
    }
    
    public function getDefiner () {
        return $this->_meta_inf['DEFINER'];
    }
    
    public function getCharsetClient () {
        return $this->_meta_inf['CHARACTER_SET_CLIENT'];
    }
    
    public function getCollationConnection () {
        return $this->_meta_inf['COLLATION_CONNECTION'];
    }
    
    public function getDatabaseCollation () {
        return $this->_meta_inf['DATABASE_COLLATION'];
    }
    
	/**
     * -------------------------------------------------------------------------------------------------------------------------------------
     * Reflector methods
     */
    
    public static function export ($routinename, PDO $pdo, $return = false) {
        $routine = new self ($routinename, $pdo);
        if ($return) {
            return (string)$routine;
        }
        else {
            echo $routine;
        }
    }
     
    public function __toString () {
        return "Routine [  routine {$this->getName()} ]";
    }
}