<?php

class ReflectionColumn implements Reflector {
    
    protected $_pdo;
    
    protected $_meta_inf;
    
    public function __construct ($columnname, PDO $pdo) {
        if ($pdo === null)
            throw new InvalidArgumentException("Second argument must be a valid PDO instance");
        
        $this->_pdo = $pdo;
        
        if (is_string($columnname)) {
            if (strpos($columnname, '`') !== false) $columnname = str_replace('`', '', $columnname);
            list($schema,$table,$column) = explode('.', $columnname);
        }
        elseif (is_array($columnname)) {
            list($schema,$table,$column) = $columnname;
        }
        else {
            throw new InvalidArgumentException("First argument must be string or array, " . gettype($tablename) . " given");
        }
        
        $query = "SELECT `TABLE_CATALOG`,`TABLE_SCHEMA`,`TABLE_NAME`,`COLUMN_NAME`,`ORDINAL_POSITION`,`COLUMN_DEFAULT`,`IS_NULLABLE`,".
        		 "`DATA_TYPE`,`CHARACTER_MAXIMUM_LENGTH`,`CHARACTER_OCTET_LENGTH`,`NUMERIC_PRECISION`,`NUMERIC_SCALE`,".
        		 "`CHARACTER_SET_NAME`,`COLLATION_NAME`,`COLUMN_TYPE`,`COLUMN_KEY`,`EXTRA`,`PRIVILEGES`,`COLUMN_COMMENT` ".
        		 "FROM `information_schema`.`COLUMNS` WHERE `TABLE_SCHEMA`=:schema AND `TABLE_NAME`=:table AND ".
        		 "`COLUMN_NAME`=:column";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $schema, PDO::PARAM_STR);
        $stmt->bindParam(':table',  $table,  PDO::PARAM_STR);
        $stmt->bindParam(':column', $column, PDO::PARAM_STR);
        if ($stmt->execute()) {
            if (!$stmt->rowCount())
                throw new ReflectionException("Column `$schema`.`$table`.`$column` does not exist");
            
            $this->_meta_inf = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    /**
     * -------------------------------------------------------------------------------------------------------------------------------------
     * Meta inf getters
     */
    
    public function getCatalog () {
        return $this->_meta_inf['TABLE_CATALOG'];
    }
    
    public function getName () {
        return $this->_meta_inf['COLUMN_NAME'];
    }
    
    public function getOrdinalPosition () {
        return $this->_meta_inf['ORDINAL_POSITION'];
    }
    
    public function getDefault () {
        return $this->_meta_inf['COLUMN_DEFAULT'];
    }
    
    public function isNullable () {
        return $this->_meta_inf['IS_NULLABLE'] == "YES";
    }
    
    public function getDataType () {
        return $this->_meta_inf['DATA_TYPE'];
    }
    
    public function getCharacterMaxLength () {
        return $this->_meta_inf['CHARACTER_MAXIMUM_LENGTH'];
    }
    
    public function getCharacterOctetLength () {
        return $this->_meta_inf['CHARACTER_OCTET_LENGTH'];
    }
    
    public function getNumericPrecision () {
        return $this->_meta_inf['NUMERIC_PRECISION'];
    }
    
    public function getNumericScale () {
        return $this->_meta_inf['NUMERIC_SCALE'];
    }
    
    public function getCharset () {
        return $this->_meta_inf['CHARACTER_SET_NAME'];
    }
    
    public function getCollation () {
        return $this->_meta_inf['COLLATION_NAME'];
    }
    
    public function getType () {
        return $this->_meta_inf['COLUMN_TYPE'];
    }
    
    public function getKey () {
        return $this->_meta_inf['COLUMN_KEY'];
    }
    
    public function getExtra () {
        return $this->_meta_inf['EXTRA'];
    }
    
    public function getPrivileges () {
        return $this->_meta_inf['PRIVILEGES'];
    }
    
    public function getComment () {
        return $this->_meta_inf['COLUMN_COMMENT'];
    }
    
    public function getBase () {
        return new ReflectionBase($this->_meta_inf['TABLE_SCHEMA'], $this->_pdo);
    }
    
    public function getTable () {
        return new ReflectionTable(array($this->_meta_inf['TABLE_SCHEMA'], $this->_meta_inf['TABLE_NAME']), $this->_pdo);
    }

    
	/**
     * -------------------------------------------------------------------------------------------------------------------------------------
     * Reflector methods
     */
    
    public static function export ($columnname, PDO $pdo, $return = false) {
        $column = new self ($columnname, $pdo);
        if ($return) {
            return (string)$column;
        }
        else {
            echo $column;
        }
    }
     
    public function __toString () {
        return "Column [  column {$this->getName()} ] { type : {$this->getType()} , pos : {$this->getOrdinalPosition()} }";
    }
}