<?php

/**
 * Reflection Base class
 *
 * @author Delespierre
 * @version 1.0.0
 * @package Reflection
 */
class ReflectionBase implements Reflector {

    /**
     * PDO Instance
     * @var PDO
     */
    protected $_pdo;
    
    /**
     * Base meta informations
     * @var array
     */
    protected $_meta_inf;
    
    /**
     * Default constructor
     * @param string $dbname
     * @param PDO $pdo
     * @throws InvalidArgumentException
     * @throws ReflectionException
     * @throws RuntimeException
     */
    public function __construct ($dbname, PDO $pdo) {
        if ($pdo === null)
            throw new InvalidArgumentException("Second argument must be a valid PDO instance");
        
        $this->_pdo = $pdo;
        
        $query = "SELECT `CATALOG_NAME`,`SCHEMA_NAME`,`DEFAULT_CHARACTER_SET_NAME`,`DEFAULT_COLLATION_NAME`,`SQL_PATH`".
                 " FROM `information_schema`.`SCHEMATA` WHERE `SCHEMA_NAME`=:dbname";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':dbname', $dbname, PDO::PARAM_STR);
        if ($stmt->execute()) {
            if (!$stmt->rowCount())
                throw new ReflectionException("Base $dbname does not exist");
            
            $this->_meta_inf = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    /**
     * Get base's catalog
     * @return string
     */
    public function getCatalog () {
        return $this->_meta_inf['CATALOG_NAME'];
    }
    
    /**
     * Get schema's name
     * @return string
     */
    public function getName() {
        return $this->_meta_inf['SCHEMA_NAME'];
    }
    
    /**
     * Get Default charset
     * @return string
     */
    public function getDefaultCharset () {
        return $this->_meta_inf['DEFAULT_CHARACTER_SET_NAME'];
    }
    
    /**
     * Get default collation
     * @return string
     */
    public function getDefaultCollation () {
        return $this->_meta_inf['DEFAULT_COLLATION_NAME'];
    }
    
    /**
     * Get SQL path
     * @return string
     */
    public function getSQLPath () {
        return $this->_meta_inf['SQL_PATH'];
    }
    
    /**
     * Get the given table in current base
     * @param string $tablename
     * @return ReflectionTable
     */
    public function getTable ($tablename) {
        return new ReflectionTable (array($this->getName(), $tablename), $this->_pdo);
    }
    
    /**
     * Get schema's tables.
     * Returns an array of ReflectionTable instances.
     * @return array
     * @throws RuntimeException
     */
    public function getTables () {
        $query = "SELECT `TABLE_SCHEMA`,`TABLE_NAME` FROM `information_schema`.`TABLES` WHERE `TABLE_SCHEMA`=:schema";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $this->_meta_inf['SCHEMA_NAME'], PDO::PARAM_STR);
        if ($stmt->execute()) {
            $list = array();
            foreach ($stmt as $row) {
                $list[] = new ReflectionTable($row, $this->_pdo);
            }
            return $list;
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    /**
     * Get the given view in the current base
     * @param string $viewname
     * @return ReflectionView
     */
    public function getView ($viewname) {
        return new ReflectionView(array($this->getName, $viewname), $this->_pdo);
    }
    
    /**
     * Get the current schema's views.
     * Return an array consisting of ReflectionView instances.
     * @return array
     * @throws RuntimeException
     */
    public function getViews () {
        $query = "SELECT `TABLE_SCHEMA`,`TABLE_NAME` FROM `information_schema`.`VIEWS` WHERE `TABLE_SCHEMA`=:schema";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $this->_meta_inf['SCHEMA_NAME'], PDO::PARAM_STR);
        if ($stmt->execute()) {
            $list = array();
            foreach ($stmt as $row) {
                $list[] = new ReflectionView($row, $this->_pdo);
            }
            return $list;
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    /**
     * Get the given routine
     * @param string $routinename
     * @return ReflectionRoutine
     */
    public function getRoutine ($routinename) {
        return new ReflectionRoutine(array($this->getName(), $routinename), $this->_pdo);
    }
    
    /**
     * Get the current schema's routines.
     * Return an array consisting of ReflectionRoutine instances.
     * @return array
     * @throws RuntimeException
     */
    public function getRoutines () {
        $query = "SELECT `ROUTINE_SCHEMA`,`ROUTINE_NAME` FROM `information_schema`.`ROUTINES` WHERE `ROUTINE_SCHEMA`=:schema";
        
        $stmt = $this->_pdo->prepare($query);
        $stmt->bindParam(':schema', $this->_meta_inf['SCHEMA_NAME'], PDO::PARAM_STR);
        if ($stmt->execute()) {
            $list = array();
            foreach ($stmt as $row) {
                $list[] = new ReflectionRoutine($row, $this->_pdo);
            }
            return $list;
        }
        else {
            $info = $this->_pdo->errorInfo();
            throw new RuntimeException ("PDOStatement execution failed: {$info[2]} with error code {$info[1]}");
        }
    }
    
    /**
     * -------------------------------------------------------------------------------------------------------------------------------------
     * Reflector methods
     */
    
    /**
     * Export the given shcema as string.
     * If the return parameters is left to false, the result will
     * be printed an no value is returned
     * @param string $dbname
     * @param PDO $pdo
     * @param boolean $return
     * @return string
     */
    public static function export ($dbname, PDO $pdo, $return = false) {
        $base = new self ($dbname, $pdo);
        if ($return) {
            return (string)$base;
        }
        else {
            echo $base;
        }
    }
    
    /**
     * (non-PHPdoc)
     * @see Reflector::__toString()
     */
    public function __toString () {
        $tables   = $this->getTables();
        $views    = $this->getViews();
        $routines = $this->getRoutines();
        return "Base [  database {$this->getName()} ] {".
        	   " - Tables [".count($tables)."] { " . implode(' ', $tables) . " } ".
               " - Views [".count($views)."] { " . implode(' ', $views) . " } ".
               " - Routines [".count($routines)."] { " . implode(' ', $routines) . " } } ";
    }
}