package xstream_reference;

import java.util.Hashtable;

import org.apache.sling.commons.classloader.DynamicClassLoaderManager;
import org.apache.sling.commons.classloader.impl.DynamicClassLoaderManagerFactory;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.SynchronousBundleListener;
import org.osgi.service.packageadmin.PackageAdmin;
import org.osgi.util.tracker.ServiceTracker;

import com.bnpp.xstream.intf.IObjectFactory;

/**
 * The activator class controls the plug-in life cycle
 */
public class Activator extends AbstractUIPlugin implements
		SynchronousBundleListener, BundleActivator {

	// The plug-in ID
	public static final String PLUGIN_ID = "xstream_reference";

	// The shared instance
	private static Activator plugin;

	private IObjectFactory factory = null;

	public IObjectFactory getFactory() {
		return factory;
	}

	/** Package admin service name */
	private static String PACKAGE_ADMIN_NAME = PackageAdmin.class.getName();

	/** A service tracker for the package admin. */
	private ServiceTracker packageAdminTracker;

	/** The service registration for the dynamic class loader manager. */
	private ServiceRegistration serviceReg;

	public ServiceRegistration getServiceReg() {
		return serviceReg;
	}

	/** The dynamic class loader service factory. */
	private DynamicClassLoaderManagerFactory service;

	public DynamicClassLoaderManagerFactory getService() {
		return service;
	}

	/** The bundle context. */
	private BundleContext bundleContext;

	public static Activator getDefault() {
		return plugin;
	}

	/**
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		plugin = this;
		this.bundleContext = context;

		this.packageAdminTracker = new ServiceTracker(this.bundleContext,
				PACKAGE_ADMIN_NAME, null);
		this.packageAdminTracker.open();

		// register service
		this.registerManagerFactory();
		this.bundleContext.addBundleListener(this);
		loadExtensions();
	}

	/**
	 * Register the dynamic class loader manager factory.
	 */
	protected void registerManagerFactory() {
		final Hashtable<String, String> props = new Hashtable<String, String>();
		props.put(Constants.SERVICE_DESCRIPTION,
				"Apache Sling Dynamic Class Loader Service");
		props.put(Constants.SERVICE_VENDOR, "The Apache Software Foundation");
		this.service = new DynamicClassLoaderManagerFactory(this.bundleContext,
				(PackageAdmin) this.packageAdminTracker.getService());
		this.serviceReg = this.bundleContext.registerService(
				new String[] { DynamicClassLoaderManager.class.getName() },
				service, props);
	}

	public ClassLoader getDynamicClassLoader() {
		try {
			DynamicClassLoaderManager manager = (DynamicClassLoaderManager) service
					.getService(this.getBundle(), getServiceReg());
			return manager.getDynamicClassLoader();
		} catch (Exception ex) {
			MessageDialog.openError(new Shell(), "Dynamic",
					"Cannot create dynamic class loader");
		}
		return null;
	}

	/**
	 * Unregister the dynamic class loader manager factory.
	 */
	protected void unregisterManagerFactory() {
		if (this.serviceReg != null) {
			this.serviceReg.unregister();
			this.serviceReg = null;
		}
		if (this.service != null) {
			this.service = null;
		}
	}

	/**
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		context.removeBundleListener(this);
		this.unregisterManagerFactory();
		if (this.packageAdminTracker != null) {
			this.packageAdminTracker.close();
			this.packageAdminTracker = null;
		}
		this.bundleContext = null;
		plugin=null;
		super.stop(context);
	}

	/**
	 * @see org.osgi.framework.BundleListener#bundleChanged(org.osgi.framework.BundleEvent)
	 */
	public void bundleChanged(BundleEvent event) {
		final boolean reload;
		if (event.getType() == BundleEvent.RESOLVED) {
			reload = this.service.isBundleUsed(event.getBundle().getBundleId())
					|| this.service.hasUnresolvedPackages(event.getBundle());
		} else if (event.getType() == BundleEvent.UNRESOLVED) {
			reload = this.service.isBundleUsed(event.getBundle().getBundleId());
		} else {
			reload = false;
		}
		if (reload) {
			this.unregisterManagerFactory();
			this.registerManagerFactory();
		}
	}

	private static final String EXTENSION_ID = "com.bnpp.xstream.factory";

	private void loadExtensions() {
		IConfigurationElement[] config = Platform.getExtensionRegistry()
				.getConfigurationElementsFor(EXTENSION_ID);
		if (config != null && config.length > 0) {
			try {
				for (IConfigurationElement e : config) {
					final Object o = e.createExecutableExtension("class");
					if (o instanceof IObjectFactory) {
						this.factory = (IObjectFactory) o;
					}
				}
			} catch (CoreException ex) {
			}
		}
	}
}
