# -*- coding: utf-8 -*-
"""
Copyright 2009 Vincent Maillol Benoît Gaëtan

This file is part of mbg Sqlite.

mbg Sqlite is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

mbg Sqlite is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with mbg Sqlite.  If not, see <http://www.gnu.org/licenses/>.
"""
from platform import python_version
is_version_3 = python_version().startswith( '3' )

if is_version_3 :
    from tkinter import Text, StringVar, TclError
else :
    from Tkinter import Text, StringVar, TclError    

import re

try :
    import otherWidget.arbre as arbre
except :
    import arbre

try :
    import otherWidget.infobull as infobull
except :
    import infobull


class VirtualConsol( Text ) :
    """
    Widget Text simulant le comportement d'un shell ou d'une console.
    """
    def __init__( self, master=None, invite1 = ">>> ",
                  invite2 = "... ", command = None, comment = '#',
                  kw_command = {}, kw_ignorcase = False , cnf={}, **kw ) :

        Text.__init__( self, master, cnf, **kw )

       
        self.tree = arbre.create_tree( kw_command )

        self.invite1 = invite1
        self.invite2 = invite2
        self.invite = self.invite1
        self.kw_command = kw_command
        self.command = command
        self.kw_ignorcase = kw_ignorcase

        self.info_mots_possible = StringVar( self )
        self.infobull = infobull.InfobullTextCursor( self, self.info_mots_possible )

        self.bind( "<KeyPress>",  self.color_text )
        self.bind( "<KeyRelease>",self.color_text )
        self.bind( "<BackSpace>", self.backspace )
        self.bind( "<Key-Return>",self.execute )
        self.bind( "<Key-Tab>",   self.autocomplement )
        self.bind( "<Key-Left>",  self.left   )
        self.bind( "<Key-Up>",    self.rappelerLigne  )
        self.bind( "<Key-Down>",  self.rappelerLigne  )
        self.bind( '<1>',         self.button_1 )
        self.bind( "<B1-Motion>", self.button_1 )
        self.bind( "<B1-Leave>", lambda event : "break" )
        self.bind( "<Double-1>", lambda event : "break" ) 
        self.bind( "<Triple-1>", lambda event : "break" )
        self.bind( "<<Paste>>" , self.insert_query )
        self.bind( "<KeyPress>", lambda event : self.infobull.withdraw(), "+" )

        self.insert( "end", self.invite )
        self.start_query = '1.0 + %d chars' % len( self.invite1 )
        self.lines = [] # Contiendra les lignes saisies pour les rappeler en cas de <Up> 
        self.num_line = 0

        self.tag_configure( 'number',  foreground = 'red' )
        self.tag_configure( 'string',  foreground = 'green' )
        self.tag_configure( 'keyword', foreground = 'purple')
        self.tag_configure( 'comment', foreground = '#C0C0C0')
        self.tag_configure( 'error'  , foreground = 'red')

        re_kw = '' # Regex pour les mots clefs
        for kw in self.kw_command :
            re_kw += r'\b%s\b|' % kw
        re_kw = re_kw[:-1]

        self.regexs = { 'number' : re.compile( r'\b\d+\.?\d*\b' ),
                        'string' : re.compile( r"'[^'\n]*(\\.[^'\n]*)*'?" ),
                       'comment' : re.compile( r'%s[^\n]*' % comment )}

        if kw_ignorcase :
            self.regexs[ 'keyword' ] = re.compile( re_kw, 2 )
        else :
            self.regexs[ 'keyword' ] = re.compile( re_kw )


    def autocomplement( self, event ):
        text = self.get()
        if text != '\n' :
            debut = text.split()[-1]
            if self.kw_ignorcase :
                fin = self.tree.get( debut.upper() )
            else :
                fin = self.tree.get( debut )
                
            if fin is not None :
                if debut.isupper() :
                    self.insert( 'end', fin )
                else :
                    self.insert( 'end', fin.lower() )
                    
            else :
                if self.kw_ignorcase :
    
                    listeDesMotPossible = [ mot for mot in self.kw_command if mot.startswith( debut.upper() ) and mot != debut.upper() ]
                else :
                    listeDesMotPossible = [ mot for mot in self.kw_command if mot.startswith( debut ) and mot != debut ]
                    
                if listeDesMotPossible != [] :
                    label = str( listeDesMotPossible )
                    label = label[1:-1] # Pour enlever les [ et ] de la liste
                    self.info_mots_possible.set( label )
                    self.infobull.show_label()
                    
        #self.insert( 'end', '\t' )
        return "break"


    def backspace( self, event ) :
        """
        Efface le caractère derrière le curseur seulement
        s'il a été écrit après la dernière insertion de self.execute. 
        """
        if int( self.index( "insert" ).split(".")[1] ) > len( self.invite ) :
             self.delete('insert - 1 chars', 'insert')
        return "break"


    def button_1( self, event ) :
        """
        Empêche de placer le curseur sur du texte déjà interprété.
        """
        last_index = self.index( "insert" )
        
        def testPosition( index ):
            if self.compare( "insert", "<=", "%d.%d" % ( float( self.index( "end" ) ), len( self.invite ) ) ) :
                self.mark_set( "insert", index )
     
        self.after_idle( testPosition, last_index )
        
        
    def color_text( self, event ) :
        """
        Coloration syntaxique. 
        """
        self.tag_remove( 'number',  self.start_query, 'end' )
        self.tag_remove( 'comment', self.start_query, 'end' )
        self.tag_remove( 'string',  self.start_query, 'end' )
        self.tag_remove( 'keyword', self.start_query, 'end' )


        text = self.get( self.start_query, 'end')

        for tag, regex in self.regexs.items() :
            lastPos = 0
            while 1 :
                match = regex.search( text, lastPos )
                if match is None :
                    break

                lastPos = match.end()
                self.tag_add( tag,
                             '%s + %d chars' % ( self.start_query, match.start() ),
                             '%s + %d chars' % ( self.start_query, lastPos ) )


    def execute( self, event, *args ) :
        """
        Execute self.command qui doit retourner un tuple composé d'un entier et d'une chaine.
        En fonction de la valeur de l'entier ( 1, 2, 0 ), self.execute affiche la chaine,
        change l'invite de commande ou
        affiche la chaine en rouge.
        *args permet le passage d'arguments à self.command.
        """

        query = self.get()[:-1]

        # Enregistre la ligne si elle n'ai pas vide et ne vient pas juste d'être inscrite.
        if self.lines == [] or self.lines[-1] != query and query != "" :
            self.lines.append( query )
            self.num_line = len( self.lines )
            
        reponse = self.command( query, *args )

        if reponse[0] == 1 : # Requête normale.
            self.invite = self.invite1
            self.insert( "end", "\n" + reponse[1] + self.invite )
            self.event_generate('<<query_interpreted>>')

        elif reponse[0] == 2 : # Changer l'invite.
            self.invite = self.invite2
            self.insert( "end", "\n" + self.invite )

        elif reponse[0] == 0 : # Error.
            self.invite = self.invite1
            self.insert( "end", "\n" + reponse[1] , 'error' )
            self.insert( "end", self.invite )
            self.event_generate('<<query_error>>')

        self.mark_start_query()
        self.see( "end" )
        self.mark_set( "insert", "end" )
        return "break"

    def insert_query( self, query ) :
        """
        Permet d'inserer une commande écrite sur
        plusieurs lignes.
        Insert_query est reliée à l'événement <<Paste>> dans ce cas,
        query est de type Event.
        """
        if type( query ) is not str :
            try :
                query = self.clipboard_get()
            except TclError as e : # Le press Papier est vide.
                query = "" 
                
        for line in query.splitlines() :
            self.insert("end", line )
            self.execute( None )
            
        return "break"

    def left( self, event ) :
        """
        Recule le curseur tant qu'il se trouve sur du texte écrit après
        la dernière insertion de self.execute. 
        """
        if int( self.index( "insert" ).split(".")[1] ) <= len( self.invite ) :
            return "break"


    def mark_start_query( self ):
        """
        Retient la position du curseur après le dernier insert. 
        """
        self.start_query = self.index( 'end - 1 chars' )


    def get( self, index1 = None, index2 = None ):
            if index1 is None :
                query = Text.get( self, self.start_query, 'end' )
                return( query )
            else :
                return( Text.get( self, index1, index2 ) )


    def rappelerLigne( self, event ) :
        """
        Rappelle les dernières lignes inscrites <Up> remonte dans les lignes les plus anciennes
        <Down> fait l'inverse. 
        """
        if self.num_line != [] :
            if event.keysym == "Up" :
                self.num_line -= 1 
            else :
                self.num_line += 1
                
            if self.num_line < 0 :
                self.num_line = 0
            elif self.num_line > len( self.lines ) -1 :
                self.num_line = len( self.lines ) -1
            else :
                self.delete( self.index( "end" ) + " - 1 line", "end" )
                self.insert( "end", "\n" + self.invite + self.lines[ self.num_line ] )
        return "break"


if __name__ == '__main__' :
    if is_version_3 :
        from tkinter import *
    else :
        from Tkinter import *
        
    def foo( query ) :
        if query == "when" :
            return 1, '27 07 2009\n'

        elif query == "who" :
            return 1, 'Maillol Vincent\n'

        elif query == "1 + 1 = 3" :
            return 0, "it's false\n"

        else :
            return 2, None

    root = Tk()
    v = VirtualConsol( root, command = foo, kw_command = ('when', 'who' ) )
    v.pack()
    #v.insert_query("who\nwhen")
    root.mainloop()
