# -*- coding: utf-8 -*-
"""
Copyright 2009 Vincent Maillol Benoît Gaëtan

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
import sqlite3

from platform import python_version
is_version_3 = python_version().startswith( '3' )

import pickle

if is_version_3 :
    from tkinter import Tk, Text, PanedWindow, Menu, StringVar, Listbox, Scrollbar, Button, Toplevel
    import tkinter.messagebox as messagebox
    import tkinter.filedialog as filedialog
else :
    from Tkinter import Tk, Text, PanedWindow, Menu, StringVar, Listbox, Scrollbar, Button, Toplevel
    import tkMessageBox as messagebox
    import tkFileDialog as filedialog


from searchconnect import Searchconnection

from otherWidget.virtualConsol import VirtualConsol 
from otherWidget.onglet import Onglet
import otherWidget.merise as merise

from client import Client
import sqlite_parseur



class sql_edit( Toplevel ):
    """
    Toplevel pour editer la dérniere requête sql éxécuter dans
    l'onglet actif et la faire interpréter par la console de
    l'onglet actif.
    """
    def __init__( self, master, onglet_actif ):
        Toplevel.__init__( self, master )
        self.text_master = master.onglet.texts[ onglet_actif ]
        self.client     = master.onglet.client[ onglet_actif ]
        
        self.title( "mbg Sqlite Client (%s)" % onglet_actif )
        self.text   = Text(      self, height = 10, width = 80 )
        self.button = Button(    self, text='Exécuter', command = self.execute_query )
        self.scroll = Scrollbar( self, command = self.text.yview )

        self.text.insert( "end", self.client.last_query )
        self.text['yscrollcommand'] = self.scroll.set

        #GDP
        self.text.grid()
        self.button.grid()
        
    def execute_query( self ):
        """
        Injecte le contenu de self.text dans self.text_master.
        """
        query = self.text.get( '0.0', 'end' )
        self.text_master.insert_query( query )
        self.event_generate('<<query_interpreted>>' ) # Pour mettre à jour les lstbox


class Onglet_Outils( Onglet ) :
    """
    Menu en onglets contenant les différents "outil" de l'application.
    """
    def __init__( self, master = None, connection = None, labels = ('Table', 'Trigger', 'MLD') ) :
        Onglet.__init__( self, master, labels )
        self.definitions = { 'Table' : [], 'Trigger' : [] } # Contiendra le code sql des tables et triggers.

        self.connection = connection
        self.cursor = self.connection.cursor()    
        
        self.lst_tbl = Listbox( self.get_frame( 'Table' ), width = 30 )
        self.txt_tbl =    Text( self.get_frame( 'Table' ), width = 70, height = 0 )
        self.scr_lst_tbl = Scrollbar( self.get_frame( 'Table' ), command = self.lst_tbl.yview)
        self.scr_txt_tbl = Scrollbar( self.get_frame( 'Table' ), command = self.txt_tbl.yview)
        self.lst_tbl['yscrollcommand'] = self.scr_lst_tbl.set
        self.txt_tbl['yscrollcommand'] = self.scr_txt_tbl.set
        
        self.lst_trg = Listbox( self.get_frame( 'Trigger'), width = 30 )
        self.txt_trg =    Text( self.get_frame( 'Trigger'), width = 70, height = 0 )
        self.scr_lst_trg = Scrollbar( self.get_frame( 'Trigger' ), command = self.lst_trg.yview)
        self.scr_txt_trg = Scrollbar( self.get_frame( 'Trigger' ), command = self.txt_trg.yview)
        self.lst_trg['yscrollcommand'] = self.scr_lst_trg.set
        self.txt_trg['yscrollcommand'] = self.scr_txt_trg.set

        self.can_mld = merise.MLD( self.get_frame( 'MLD' ), bg = 'ivory', width = 900, scrollregion=(0, 0, "20i", "20i") )
        self.scr_x_can = Scrollbar( self.get_frame( 'MLD' ),
                                    command = self.can_mld.xview, orient="horizontal")
        self.scr_y_can = Scrollbar( self.get_frame( 'MLD' ),
                                    command = self.can_mld.yview, orient="vertical")
        self.can_mld['xscrollcommand'] = self.scr_x_can.set
        self.can_mld['yscrollcommand'] = self.scr_y_can.set

        # GDP
        self.get_frame( 'Table' ).rowconfigure(0, weight=1)
        self.scr_lst_tbl.grid( row=0, column=0, sticky = 'ns' )
        self.lst_tbl.grid(     row=0, column=1, sticky = 'ns' )
        self.scr_txt_tbl.grid( row=0, column=2, sticky = 'ns' )
        self.txt_tbl.grid(     row=0, column=3, sticky = 'ns' )
        
        self.get_frame( 'Trigger' ).rowconfigure(0, weight=1)
        self.scr_lst_trg.grid( row=0, column=0, sticky = 'ns' )
        self.lst_trg.grid(     row=0, column=1, sticky = 'ns' )
        self.scr_txt_trg.grid( row=0, column=2, sticky = 'ns' )
        self.txt_trg.grid(     row=0, column=3, sticky = 'ns' )
        
        self.get_frame( 'MLD' ).rowconfigure( 0, weight = 1 )
        self.scr_y_can.grid( row=0, column=0, sticky = 'ns' )
        self.can_mld.grid(   row=0, column=1, sticky = 'ns' )
        self.scr_x_can.grid( row=1, column=1, sticky = 'ew' )
        
        self.lst_tbl.bind( "<ButtonRelease-1>", lambda event : self.montrer_code( event, objet = 'Table' ) )
        self.lst_trg.bind( "<ButtonRelease-1>", lambda event : self.montrer_code( event, objet = 'Trigger') )
        
        self.mise_a_jour_listbox()
        self.mise_a_jour_mld()

        self.master.bind_all( '<<query_interpreted>>', self.mise_a_jour_listbox )
        self.master.bind_all( '<<query_interpreted>>', self.mise_a_jour_mld, '+')

    def mise_a_jour_listbox( self, event = None ) :
        """
        Met à jour toutes les listboxs.
        """
        # Supression des dernieres données
        self.definitions = { 'Table' : [], 'Trigger' : [] }
        
        self.lst_tbl.delete( '0', 'end' )
        self.cursor.execute( "SELECT name, sql FROM sqlite_master WHERE type = 'table'" )
        for table, sql in self.cursor.fetchall() :
            self.lst_tbl.insert( 'end', table )
            self.definitions['Table'].append( sql )

        self.lst_trg.delete( '0', 'end' )
        self.cursor.execute( "SELECT name, sql FROM sqlite_master WHERE type = 'trigger'" )
        for trigger, sql in self.cursor.fetchall() :
            self.lst_trg.insert( 'end', trigger )
            self.definitions['Trigger'].append( sql )
            
        self.cursor.close()

    def mise_a_jour_mld( self, event = None ) :

        dico = sqlite_parseur.parse( self.connection )
        self.can_mld.delete( "all" )

        entitys = {}
        pos_x = pos_y = 10
        for table, structure in dico.items():
            # Affichage de la table.
            entitys[ table ] = self.can_mld.create_entity(
                                    pos_x, pos_y,
                                    fill = 'white',
                                    head = table,
                                    columns = structure['field'],
                                    pks = structure['pks'],
                                    fks = [ fk[0] for fk in structure['fks'] ]
                                    )
            
            # Calcul de la position de la prochaine de la table.
            pos_x += self.can_mld.sizeItem( entitys[ table ] ) [ 0 ]
            pos_x += 30
            if pos_x >= 800 :
                pos_x = 10
                pos_y += max( ( self.can_mld.sizeItem( entity )[ 1 ] for entity in entitys.values() ) )               
                pos_y += 30

        # Affichage des liens entre les clefs étrangères 
        for table, structure in dico.items():
            if structure[ 'fks' ] != [] :
                fks = structure[ 'fks' ]
                for fk in fks :
                    self.can_mld.link( {
                        entitys[ table  ]  : fk[ 0 ],
                        entitys[ fk[ 1 ] ] : fk[ 2 ]} )
        

    def montrer_code( self, event, objet ):
        """
        Affiche le code sql qui a servi à la création de l'objet sélectionné. 
        """
        if is_version_3 :
            text = self.get_frame( objet ).grid_slaves( row=0, column=3 ).__next__()
        else :
            text = self.get_frame( objet ).grid_slaves( row=0, column=3 )[0]
        
        text.delete( '1.0', 'end' )
        indices = event.widget.curselection()
        if indices != () :
            text.insert( 'end',  self.definitions[ objet ][ int( indices[0] ) ] )
        

class ClientWindow( Tk ) :
    """
    Fenêtre principale de l'application 
    """
    def __init__( self, screenName=None, baseName=None, className='Tk', useTk=1, sync=0, use=None ) :
        Tk.__init__( self, screenName, baseName, className, useTk, sync, use )
        self.file_name = "old.pkl"

        self.wm_title("mbg Sqlite Client")
        self.wm_sizefrom(who="program")
        self.wm_resizable( width=False, height=True )
        
        file_ac = open( self.file_name, "rb" )
        self.anciennes_connexions = pickle.loads( file_ac.read() )
        file_ac.close()

        self.barDeMenu = Menu( self )
        self.config( menu = self.barDeMenu )
        
        self.fichierMenu = Menu(self.barDeMenu)
        self.barDeMenu.add_cascade( label = 'Fichier', menu = self.fichierMenu )

        self.connectionMenu = Menu(self.barDeMenu)
        self.barDeMenu.add_cascade( label = 'Connexion', menu = self.connectionMenu )

        self.bddMenu = Menu(self.barDeMenu)
        self.barDeMenu.add_cascade( label = 'Base de données', menu = self.bddMenu )

        self.requeteMenu = Menu(self.barDeMenu)
        self.barDeMenu.add_cascade( label = 'Requête', menu = self.requeteMenu )


        self.fichierMenu.add_command( label='Quitter', foreground='red', command=self.bye )
        
        self.connectionMenu.add_command( label='Nouvelle connexion', command=self.ouvrir_bdd )
        self.connectionMenu.add_separator()
        for con in self.anciennes_connexions :
            self.connectionMenu.add_command( label = con,
                                             command = lambda arg = con : self.add_pane( arg )  )
        self.connectionMenu.add_separator()
        self.connectionMenu.add_command( label='Fermer la connexion actuel',
                                         command=self.closeConectCurent, state="disable" )

        self.bddMenu.add_command( label='Nouvelle bdd', command=self.make_file_bdd )
        self.bddMenu.add_command( label='Créer une bdd en RAM', command=self.add_pane )

        self.requeteMenu.add_command( label='Editer dernière requête', command = self.openLastQuery )
        self.requeteMenu.add_command( label="Forcer l'interprétation", command = self.forcer_interpretation )
        
        self.onglet = Onglet( self, position = "bottom" )
        self.onglet.client = {} # Pour fermer les bdds proprement, et avoir accés à last_query
        self.onglet.texts  = {} # Pour avoir accés aux VirtualConsols.
        self.rowconfigure(  0, weight=1 )
        self.onglet.grid( row = 0, sticky="nsew" )

        self.protocol ( "WM_DELETE_WINDOW", self.bye) 


    def add_pane( self, chemin_bdd = ":memory:" ):

        if self.onglet.add_onglet( chemin_bdd ) :
            client = Client( chemin_bdd )
            self.onglet.client[ chemin_bdd ] = client
            
            text = VirtualConsol( self, invite1 = "sqlite3 --> ", invite2 = "sqlite3 ... ",
                            command = client.execute_query, comment = '--', kw_ignorcase = True, 
                            height = 20, kw_command = (
                'ABORT','ADD','AFTER','ALL','ALTER','ANALYZE','AND','AS','ASC', 'ATTACH','AUTOINCREMENT',
                'BEFORE','BEGIN','BETWEEN','BY','CASCADE','CASE','CAST','CHECK','COLLATE',
                'COLUMN','COMMIT','CONFLICT','CONSTRAINT','CREATE','CROSS','CURRENT_DATE','CURRENT_TIME',
                'CURRENT_TIMESTAMP','DATABASE','DEFAULT','DEFERRABLE','DEFERRED','DELETE',
                'DESC','DETACH','DISTINCT','DROP','EACH','ELSE','END','ESCAPE','EXCEPT',
                'EXCLUSIVE','EXISTS','EXPLAIN','FAIL','FOR','FOREIGN','FROM','FULL','GLOB',
                'GROUP','HAVING','IF','IGNORE','IMMEDIATE','IN','INDEX','INITIALLY','INNER',
                'INSERT','INSTEAD','INTERSECT','INTO','IS','ISNULL','JOIN','KEY','LEFT',
                'LIKE','LIMIT','MATCH','NATURAL','NOT','NOTNULL','NULL','OF','OFFSET',
                'ON','OR','ORDER','OUTER','PLAN','PRAGMA','PRIMARY','QUERY','RAISE','REFERENCES',
                'REGEXP','REINDEX','RENAME','REPLACE','RESTRICT','RIGHT','ROLLBACK','ROW',
                'SELECT','SET','TABLE','TEMP','TEMPORARY','THEN','TO','TRANSACTION','TRIGGER',
                'UNION','UNIQUE','UPDATE','USING','VACUUM','VALUES','VIEW','VIRTUAL','WHEN','WHERE') )

            # Injection des dernieres lignes saisies pour cette connexion.
            try :
                file_lines = open( "lines.pkl", "rb" )
                dico_lines = pickle.loads( file_lines.read() )
                if chemin_bdd in dico_lines :
                        text.lines = dico_lines[ chemin_bdd ]
                        text.num_line = len( text.lines )
                file_lines.close()
            except :
                pass
                
        
            self.onglet.texts[ chemin_bdd ] = text 
            self.onglet.get_frame( chemin_bdd ).rowconfigure(0, weight=1)
            panes = PanedWindow( self.onglet.get_frame( chemin_bdd ), orient = 'vertical',
                                showhandle = True, handlesize = 6, sashwidth = 10 )
            panes.unbind_all( "<Key-Tab>" )
            
            onglet_outil = Onglet_Outils( self, client.connection )
            panes.add( onglet_outil )
            panes.add( text )
            panes.grid( row=0, sticky = "nsew")
            
            if self.onglet.nmb_onglet() > 1 :
                self.connectionMenu.entryconfigure(
                        self.connectionMenu.index( "end" ), state = "normal" ) 
 
    def bye( self ) :
        """
        Referme toutes les connexions et se détruit.
        """
        # Sauvegarde des lignes saisies
        file_line = open( "lines.pkl", "rb" )
        dico_line = pickle.loads( file_line.read() )
        file_line.close()

        for key in self.onglet.get_onglets() :
            if key in dico_line :
                dico_line[ key ] += self.onglet.texts[ key ].lines
            else :
                dico_line[ key ] = self.onglet.texts[ key ].lines
 
        dumps_lines = pickle.dumps( dico_line )
        file_line = open( "lines.pkl", 'wb' )
        file_line.write( dumps_lines )
        file_line.close()
        
        for client in self.onglet.client.values() :
            client.close()
        self.destroy()

    def closeConectCurent( self ) :
        """
        Ferme la connexion afficher dans l'onglet actif. 
        """
        key = self.onglet.onglet_actif()
        # Sauvegarde des lignes saisies
        file_line = open( "lines.pkl", "rb" )
        dico_line = pickle.loads( file_line.read() )
        file_line.close()
        
        if key in dico_line :
            dico_line[ key ] += self.onglet.texts.pop( key ).lines
        else :
            dico_line[ key ] = self.onglet.texts.pop( key ).lines            

        dumps_lines = pickle.dumps( dico_line )
        file_line = open( "lines.pkl", 'wb' )
        file_line.write( dumps_lines )
        file_line.close()

        # Fermeture de la connexion et destruction de l'onglet.
        self.onglet.client.pop( key ).close()
        self.onglet.del_onglet( key )
        if self.onglet.nmb_onglet() <= 1 :
            self.connectionMenu.entryconfigure(
                    self.connectionMenu.index( "end" ), state = "disable")

    def forcer_interpretation( self ) :
        """
        Demander au client d'interpréter la requête.
        il arrive qu'une requête erronée contenant des "begin"
        ne veuille pas être interprétée. 
        """
        self.onglet.texts[ self.onglet.onglet_actif() ].execute( None, True )
            
    def ouvrir_bdd( self ):
        """
        Recherche une base de données dans l'arborescence de fichiers. 
        """
        chemin = filedialog.askopenfilename( filetypes= [ ('sqlite3','*.db3'), ('tous','*.*') ] )
        if chemin != '' :
            # Le fichier commence-t-il par "SQLite format 3" ? 
            file = open( chemin, 'rb' )
            entete = file.read( 15 )
            file.close()
            fichier_valide =( entete == b"SQLite format 3" )

            if not fichier_valide :
                # Proposition pour quand même ouvrir le fichier.
                fichier_valide = "yes" == messagebox.askquestion( 'mbgSqlite', "Le fichier sélectionné semble ne pas être valide.\nVoulez-vous l'ouvrir ?", icon = "warning" )

            if fichier_valide :
                try :
                    sqlite3.connect( chemin ).execute("select * from sqlite_master")
                except Exception as sqlite_msg :
                    messagebox.showerror("Erreur Sqlite3", "Sqlite3 ne peut pas ouvrir ce fichier.\n%s" % sqlite_msg )
                else :
                    # Si le chemin n'est pas déjà inscrit. 
                    if chemin not in self.anciennes_connexions :
                        # Insertion du chemin dans le fichier old.pkl.
                        self.anciennes_connexions.insert( 0, chemin )
                        a = pickle.dumps( self.anciennes_connexions )
                        file_ac = open( self.file_name, 'wb' )
                        file_ac.write( a )
                        file_ac.close()
                        # Insertion du chemin dans le menu connexion.
                        self.connectionMenu.insert_command( 3, label = chemin,
                                             command = lambda arg = chemin : self.add_pane( arg ) )
                    
                    self.add_pane( chemin )

    def make_file_bdd( self ) :

            chemin = ( filedialog.asksaveasfilename( title='Créer un nouveau fichier sqlite3',
                                                     initialfile = '*.db3',
                                                     defaultextension = ".db3" ) )
            if chemin != '' :
                self.anciennes_connexions.insert( 0, chemin )
                a = pickle.dumps( self.anciennes_connexions )
                file_ac = open( self.file_name, 'wb' )
                file_ac.write( a )
                file_ac.close()
                self.add_pane( chemin )

    def openLastQuery( self ):
        """
        Récupère la dernière requête saisie dans l'onglet courant et l'affiche dans
        un widget text pour pouvoir faire ré-interpréter la requête. 
        """
        sql_edit( self, self.onglet.onglet_actif() )

if __name__ == "__main__" :
    root = ClientWindow()
    var = StringVar()
    
    root.wm_withdraw()
    sc = Searchconnection( root, variable = var )
    root.wait_variable( var )

    sc.destroy()
    root.wm_deiconify()
    root.add_pane( var.get() )
    root.mainloop()
