#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdlib.h>
#include <assert.h>
#include "SSaveGdiObj.h"

/* Fonction de vrification.
   ------------------------- */
EXTERN_C void SSaveGdiObj_AssertValid(struct SSaveGdiObj const * pcThis)
{
	assert(pcThis != NULL);
	assert(!pcThis->m_bDestroyed);
	if(pcThis->m_hdc == NULL)
	{
		//If no device context, all GDI objects must be NULL.
		assert(pcThis->m_hFirstObject == NULL);
		assert(pcThis->m_hCurrentObject == NULL);
		assert(!pcThis->m_bDeleteSelected);
	}
	if(pcThis->m_objectType == 0)
	{
		//If no object type, all GDI objects must be NULL.
		assert(pcThis->m_hFirstObject == NULL);
		assert(pcThis->m_hCurrentObject == NULL);
		assert(!pcThis->m_bDeleteSelected);
	}
	if(pcThis->m_hFirstObject == NULL)
	{
		//If no first object, all GDI objects must be NULL.
		assert(pcThis->m_hFirstObject == NULL);
		assert(pcThis->m_hCurrentObject == NULL);
		assert(!pcThis->m_bDeleteSelected);
	}
	else
	{
		assert(GetObjectType(pcThis->m_hFirstObject) == pcThis->m_objectType);
	}
	
	if(pcThis->m_hCurrentObject == NULL)
	{
		//If no current object, m_bDeleteSelected must be FALSE, and the DC must hold the old object.
		assert(!pcThis->m_bDeleteSelected);
		assert(GetCurrentObject(pcThis->m_hdc, pcThis->m_objectType) == pcThis->m_hFirstObject);
	}
	else
	{
		//If there is an object selected, it shall not be the same as the old object
		//(because when it is the old object, the state is "nothing selected".
		assert(GetObjectType(pcThis->m_hCurrentObject) == pcThis->m_objectType);
		assert(GetCurrentObject(pcThis->m_hdc, pcThis->m_objectType) == pcThis->m_hCurrentObject);
		assert(pcThis->m_hCurrentObject != pcThis->m_hFirstObject);
	}
}


/* Fonction pour changer l'objet GDI en cours
   ------------------------------------------ */
EXTERN_C BOOL SSaveGdiObj_Select(struct SSaveGdiObj *pThis, HGDIOBJ hNewObj, BOOL bDeleteSelected)
{
	SSaveGdiObj_AssertValid(pThis);
	//If not in a OK state, always reject.
	if(pThis->m_hdc==NULL || pThis->m_objectType==0)
	{
		if(bDeleteSelected)
			DeleteObject(hNewObj);
		SSaveGdiObj_AssertValid(pThis);
		return FALSE;
	}
	//If new object is NULL, reject it and do no change.
	if(hNewObj==NULL)
		return FALSE;
	//If new object is of wrong type, reject it.
	if(GetObjectType(hNewObj)!=pThis->m_objectType)
	{
		if(bDeleteSelected)
			DeleteObject(hNewObj);
		SSaveGdiObj_AssertValid(pThis);
		return FALSE;
	}
		
	//If the new object is the same as the first one, reject it if deleting, or deselect now.
	if(hNewObj == pThis->m_hFirstObject)
	{
		HGDIOBJ hOldObject;
		if(bDeleteSelected)
			return FALSE;

		hOldObject = SelectObject(pThis->m_hdc, pThis->m_hFirstObject);
		if(hOldObject == NULL)
		{
			//Failed for an unknown reason : Reject and do no further change.
			SSaveGdiObj_AssertValid(pThis);
			return FALSE;
		}
		else
		{
			//Success : Delete old object if requested.
			assert(hOldObject == pThis->m_hCurrentObject);
			if(pThis->m_bDeleteSelected)
				DeleteObject(pThis->m_hCurrentObject);
			pThis->m_hCurrentObject = NULL; //"nothing selecte" state.
			pThis->m_bDeleteSelected = FALSE;
		}
		SSaveGdiObj_AssertValid(pThis);
		return TRUE;
	}
	
	//If the new object is currently selected, just change its delete flag
	if(hNewObj==pThis->m_hCurrentObject)
	{
		pThis->m_bDeleteSelected = bDeleteSelected;
		SSaveGdiObj_AssertValid(pThis);
		return TRUE;
	}
	//If there is currently an object selected, replace it.
	//Else, just select the new one.
	if(pThis->m_hCurrentObject != NULL)
	{
		//Replace
		HGDIOBJ hOldObject = SelectObject(pThis->m_hdc, hNewObj);
		if(hOldObject == NULL)
		{
			//Failed for an unknown reason : Reject and do no further change.
			if(bDeleteSelected)
				DeleteObject(hNewObj);
			SSaveGdiObj_AssertValid(pThis);
			return FALSE;
		}
		else
		{
			//Success : Delete old object if requested.
			assert(hOldObject == pThis->m_hCurrentObject);
			if(pThis->m_bDeleteSelected)
				DeleteObject(pThis->m_hCurrentObject);
			pThis->m_hCurrentObject = hNewObj;
			pThis->m_bDeleteSelected = bDeleteSelected;
		}
	}
	else
	{
		//Select
		HGDIOBJ hOldObject = SelectObject(pThis->m_hdc, hNewObj);
		if(hOldObject == NULL)
		{
			//Failed for an unknown reason : Reject and do no further change.
			if(bDeleteSelected)
				DeleteObject(hNewObj);
			SSaveGdiObj_AssertValid(pThis);
			return FALSE;
		}
		else
		{
			//Success.
			assert(hOldObject == pThis->m_hFirstObject);
			assert(!pThis->m_bDeleteSelected);
			pThis->m_hCurrentObject = hNewObj;
			pThis->m_bDeleteSelected = bDeleteSelected;
		}
	}
	SSaveGdiObj_AssertValid(pThis);
	return TRUE;
}

/* Fonction d'change
   ------------------ */
EXTERN_C void SSaveGdiObj_Swap(struct SSaveGdiObj *pA, struct SSaveGdiObj *pB)
{
	SSaveGdiObj_AssertValid(pA);
	SSaveGdiObj_AssertValid(pB);
	assert( !pA->m_bIsCppClass );
	assert( !pB->m_bIsCppClass );
	
	{	HDC hTmp = pA->m_hdc;
		pA->m_hdc = pB->m_hdc;
		pB->m_hdc = hTmp; }
	{	BOOL bTmp = pA->m_bDeleteSelected;
		pA->m_bDeleteSelected = pB->m_bDeleteSelected;
		pB->m_bDeleteSelected = bTmp; }
	{	DWORD tmp = pA->m_objectType;
		pA->m_objectType = pB->m_objectType;
		pB->m_objectType = tmp; }
	{	HGDIOBJ hTmp = pA->m_hFirstObject;
		pA->m_hFirstObject = pB->m_hFirstObject;
		pB->m_hFirstObject = hTmp;	}
	{	HGDIOBJ hTmp = pA->m_hCurrentObject;
		pA->m_hCurrentObject = pB->m_hCurrentObject;
		pB->m_hCurrentObject = hTmp;	}
	/* On n'essaie pas d'inverser m_bDestroyed:
	   Aprs tout, ils doivent tre faux tous les deux. */
	/* On n'essaie pas d'inverser m_bIsCppClass:
	   Aprs tout, ils doivent tre faux tous les deux. */
}

/* Constructeur vide
   ----------------- */
EXTERN_C void SSaveGdiObj_ConstructFromNothing(struct SSaveGdiObj *pThis)
{
	pThis->m_hdc = NULL;
	pThis->m_bDeleteSelected = FALSE;
	pThis->m_objectType = 0;
	pThis->m_hFirstObject = NULL;
	pThis->m_hCurrentObject = NULL;
	pThis->m_bDestroyed = FALSE;
	pThis->m_bIsCppClass = FALSE;
	
	SSaveGdiObj_AssertValid(pThis);
}

/* Constructeur  partir d'un type d'objet GDI
   ------------------------------------------- */
EXTERN_C void SSaveGdiObj_ConstructFromType(struct SSaveGdiObj *pThis, HDC hdc, DWORD objectType)
{
	pThis->m_hdc = hdc;
	pThis->m_bDeleteSelected = FALSE;
	pThis->m_objectType = objectType;
	pThis->m_hFirstObject = GetCurrentObject(hdc, objectType);
	pThis->m_hCurrentObject = NULL;
	pThis->m_bDestroyed = FALSE;
	pThis->m_bIsCppClass = FALSE;
	
	SSaveGdiObj_AssertValid(pThis);
}

/* Constructeur  partir d'un nouvel d'objet GDI
   --------------------------------------------- */
EXTERN_C BOOL SSaveGdiObj_ConstructFromObj(struct SSaveGdiObj *pThis, HDC hdc, HGDIOBJ hNewObj, BOOL bDeleteSelected)
{
	pThis->m_hdc = hdc;
	pThis->m_bDeleteSelected = FALSE;
	pThis->m_objectType = GetObjectType(hNewObj);
	pThis->m_hFirstObject = GetCurrentObject(hdc, pThis->m_objectType);
	pThis->m_hCurrentObject = NULL;
	pThis->m_bDestroyed = FALSE;
	pThis->m_bIsCppClass = FALSE;

	return SSaveGdiObj_Select(pThis, hNewObj, bDeleteSelected);
}

/* Destructeur
   ----------- */
EXTERN_C void SSaveGdiObj_Destroy(struct SSaveGdiObj *pThis)
{
	SSaveGdiObj_AssertValid(pThis);
	//If there is currently a selected object, select the old one (and destroy the current one if needed).
	if(pThis->m_hCurrentObject != NULL)
	{
		HGDIOBJ hOldObject = SelectObject(pThis->m_hdc, pThis->m_hFirstObject);
		assert(hOldObject == pThis->m_hCurrentObject);
		if(pThis->m_bDeleteSelected)
			DeleteObject(pThis->m_hCurrentObject);
		pThis->m_hCurrentObject = NULL;
		pThis->m_bDeleteSelected = FALSE;
	}
	else
	{
		//If not, either there is no DC/object type, or the DC's object is the old one.
		assert(pThis->m_hdc==NULL || pThis->m_objectType==0 || GetCurrentObject(pThis->m_hdc, pThis->m_objectType)==pThis->m_hFirstObject);
	}
	SSaveGdiObj_AssertValid(pThis);
	pThis->m_bDestroyed = TRUE;
}


/* Constructeur dynamique  partir d'un type d'objet GDI
   ----------------------------------------------------- */
EXTERN_C struct SSaveGdiObj * SSaveGdiObj_NewFromType(HDC hdc, DWORD objectType)
{
	struct SSaveGdiObj *pThis = malloc(sizeof *pThis);
	if(pThis)
		SSaveGdiObj_ConstructFromType(pThis, hdc, objectType);
	return pThis;
}

/* Constructeur dynamique  partir d'un nouvel d'objet GDI
   ------------------------------------------------------- */
EXTERN_C struct SSaveGdiObj * SSaveGdiObj_NewFromObj(HDC hdc, HGDIOBJ hNewObj, BOOL bDeleteSelected)
{
	struct SSaveGdiObj *pThis = malloc(sizeof *pThis);
	if(pThis)
	{
		BOOL res = SSaveGdiObj_ConstructFromObj(pThis, hdc, hNewObj, bDeleteSelected);
		if(!res)
		{
			//Failed : free and return NULL.
			free(pThis), pThis=NULL;
		}
	}
	return pThis;
}

/* Destructeur dynamique
   --------------------- */
EXTERN_C void SSaveGdiObj_Delete(struct SSaveGdiObj *pThis)
{
	if(pThis==NULL)
		return;
	assert( ! pThis->m_bIsCppClass );
	SSaveGdiObj_Destroy(pThis);
	free(pThis);
}

