package net.library.jiga.sf3;

//
//  Sprite.java
//  Sf3JSwing
//
//  Created by website on 08.08.06.
//  Copyright 2006 __MyCompanyName__. All rights reserved.
//
import com.sun.media.imageio.stream.FileChannelImageInputStream;
import com.sun.media.imageio.stream.FileChannelImageOutputStream;
import java.awt.*;
import java.awt.image.*;
import java.awt.geom.*;
import java.awt.image.renderable.ParameterBlock;
import java.lang.ref.PhantomReference;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.net.URI;
import java.util.Formatter;
import java.util.Iterator;
import java.util.Map;
import javax.imageio.IIOImage;
import javax.imageio.ImageReader;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.ImageWriter;
import javax.imageio.event.IIOReadProgressListener;
import javax.imageio.event.IIOReadWarningListener;
import javax.imageio.event.IIOWriteProgressListener;
import javax.imageio.event.IIOWriteWarningListener;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;
import javax.media.jai.GraphicsJAI;
import javax.swing.JComponent;
import java.io.*;
import javax.imageio.ImageIO;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import javax.media.jai.JAI;
import javax.media.jai.PerspectiveTransform;
import javax.media.jai.RenderedOp;
import javax.media.jai.WarpPerspective;
import net.library.jiga.sf3.system.CoalescedThreadsMonitor;
import net.library.jiga.sf3.system.Resource;
import net.library.jiga.sf3.system.SfMediaTracker;
import net.library.jiga.sf3.system.SpritesCacheManager;
import net.library.jiga.sf3.system.Threaded;

/**
 * Sprites used in Java Environnement either as a lightweight component or as a painter on specified ImageObserver. Roughly uses BufferedImage instances to handle pictures of any Java supported formats.
 * (e.g. PNG images, JPEG compression, BMP bitmaps)
 * @discussion  BufferedImage type used is TYPE_4BYTE_ABGR supporting translucide images (PNG sprites, generally)
 */
public class Sprite extends JComponent implements CompositeCapable, Serializable, Cloneable, Resource, Threaded {

    /** serial version UID */
    private static final long serialVersionUID = 2323;
    /** ImageObserver to avoid Garbage flushing
     * protected Component obs;*/
    protected transient Component obs = this;
    /** base URI link */
    protected URI base;
    /** src */
    protected transient Object src = null;
    /** Image data cache */
    private transient Image data;
    /** reference to data */
    private transient PhantomReference<? extends Image> ref;
    /** reference Queue */
    private transient ReferenceQueue<? extends Image> refQueue = new ReferenceQueue<Image>();
    /** cache type
     * @see #VOLATILE
     * @see #BUFFERED */
    protected int cache;
    /** hashCode to identify the sprite*/
    protected int hash;
    /** rendering hint */
    //protected transient RenderingHints rendering = new RenderingHints(null);
    /***/
    protected int trackerPty = hash;
    /* media tracker */
    protected transient MediaTracker mt;
    /** Dimensions */
    protected Dimension size;
    /** Volatile cache */
    public static final int VOLATILE = 0;
    /** Buffered cache */
    public static final int BUFFERED = 1;

    /***/
    public void setMt(MediaTracker mt, Component pobs) {
        obs = pobs;
        if (this.mt instanceof MediaTracker) {
            if (!this.mt.equals(mt)) {
                if (load) {
                    final CoalescedThreadsMonitor monitor = imageSynch;
                    synchronized (monitor.getMonitor(false)) {
                        System.err.println("MediaTracker changed (" + this.mt.hashCode() + " to " + mt.hashCode() + ") for " + base);
                        MediaTracker previous = this.mt;
                        this.mt = mt;
                        if (data instanceof Image) {
                            trackImage(trackerPty, data, size);
                        }
                        if (data instanceof Image) {
                            previous.removeImage(data);
                        }
                        invalidate();
                        monitor.notifyOnMonitor();
                    }
                } else {
                    this.mt = mt;
                }
            }
        } else {
            this.mt = mt;
            if (load) {
                final CoalescedThreadsMonitor monitor = imageSynch;
                synchronized (monitor.getMonitor(false)) {
                    if (data instanceof Image) {
                        trackImage(trackerPty, data, size);
                    }
                    invalidate();
                    monitor.notifyOnMonitor();
                }
            }
        }
    }

    /***/
    public MediaTracker getMt() {
        return mt;
    }

    /***/
    public void setTrackerPty(int trackerPty) {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (data instanceof Image && trackerPty != this.trackerPty) {
                invalidate();
                System.err.println("Tracker Pty changed (" + this.trackerPty + " to " + trackerPty + ") for " + base);
                trackImage(trackerPty, data, size);
                mt.removeImage(data, this.trackerPty);
                this.trackerPty = trackerPty;
            } else if (this.trackerPty != trackerPty) {
                invalidate();
                this.trackerPty = trackerPty;
            }
            monitor.notifyOnMonitor();
        }
    }

    /***/
    public int getTrackerPty() {
        return trackerPty;
    }
    /** default type */
    public static final int DEFAULT_TYPE = BufferedImage.TYPE_4BYTE_ABGR;
    /** buffered type @default BufferedImage.TYPE_4BYTE_ABGR */
    public int _type = DEFAULT_TYPE;
    /** axial symetrics HORIZONTAL */
    public static final int HORIZONTAL = 0;
    /** axial symetrics VERTICAL */
    public static final int VERTICAL = 1;
    /** axial symetrics DIAGONAL POSITIVE */
    public static final int DIAGONAL_POS = 2;
    /** axial symetrics DIAGONAL NEGATIVE */
    public static final int DIAGONAL_NEG = 3;
    /** none constant */
    public static final int NONE = -1;
    /** actual zoom @default 1.0*/
    protected double zoom = 1.0;
    /** actual mirror orientation */
    protected int mirror = NONE;
    /** disk cached */
    protected boolean diskCached;
    /** image mime type */
    protected String mime;
    /***/
    protected transient PerspectiveTransform currentPrePerspective;
    /***/
    protected transient AffineTransform currentPreTransforming;
    /***/
    protected transient AffineTransform currentPreMirroring;
    /***/
    protected transient AffineTransform currentPreScaling;
    /***/
    protected PerspectiveTransform perspective = new PerspectiveTransform();
    /** image transforms @default is identity matrix*/
    protected AffineTransform transforming = new AffineTransform();
    /***/
    protected AffineTransform mirroring = new AffineTransform();
    /***/
    protected AffineTransform scaling = new AffineTransform();
    /**
     * innerResource mode (inner(true)/outer(false) .jar) @default false
     */
    protected boolean innerResource = false;
    /***/
    protected transient Set<IIOReadProgressListener> rProgList = Collections.synchronizedSet(new HashSet<IIOReadProgressListener>());
    /***/
    protected transient Set<IIOWriteProgressListener> wProgList = Collections.synchronizedSet(new HashSet<IIOWriteProgressListener>());
    /***/
    protected transient Set<IIOReadWarningListener> rWarnList = Collections.synchronizedSet(new HashSet<IIOReadWarningListener>());
    /***/
    protected transient Set<IIOWriteWarningListener> wWarnList = Collections.synchronizedSet(new HashSet<IIOWriteWarningListener>());
    /***/
    private boolean load = true;
    /***/
    private int MAX_SIZE = 4000000;
    /***/
    protected String image_dir = "cache" + File.separator + "imageIO";
    /***/
    protected boolean compositeEnabled = false;
    /***/
    protected transient Paint pnt = null;
    /***/
    protected transient Color clr = null;
    /***/
    protected transient Composite cps = null;
    /***/
    private transient boolean composited;
    /***/
    protected transient CoalescedThreadsMonitor paintMonitor;
    /***/
    protected transient boolean painting = false;
    /***/
    protected transient CoalescedThreadsMonitor packMonitor;
    /***/
    protected transient boolean packing = false;
    /***/
    protected transient CoalescedThreadsMonitor validateMonitor;
    /***/
    protected transient boolean validating = false;
    /***/
    protected transient CoalescedThreadsMonitor imageSynch;
    /***/
    protected HashMap<String, Serializable> attributes = new HashMap<String, Serializable>();

    /***/
    private void loadDefaultAttributes(HashMap<String, Serializable> attributes) {
        attributes.put("composited", false);
        attributes.put("currentPreTransforming", new AffineTransform());
        attributes.put("currentPreMirroring", new AffineTransform());
        attributes.put("currentPreScaling", new AffineTransform());
        attributes.put("currentPrePerspective", new PerspectiveTransform());
        setEnabled(false);
        resetAttributes();
    }

    /***/
    private void storeCurrentAttributes() {
        attributes.put("composited", composited);
        attributes.put("currentPreTransforming", currentPreTransforming);
        attributes.put("currentPreMirroring", currentPreMirroring);
        attributes.put("currentPreScaling", currentPreScaling);
        attributes.put("currentPrePerspective", currentPrePerspective);
    }

    /***/
    private HashMap<String, Serializable> cloneAttributes() {
        Map<String, Serializable> baseAttr = Collections.synchronizedMap(this.attributes);
        HashMap<String, Serializable> attributes = (HashMap<String, Serializable>) this.attributes.clone();
        // add cloneables
        synchronized (baseAttr) {
            for (Iterator<String> i = baseAttr.keySet().iterator(); i.hasNext();) {
                String k = i.next();
                if (baseAttr.get(k) instanceof AffineTransform) {
                    attributes.put(k, (Serializable) ((AffineTransform) baseAttr.get(k)).clone());
                } else if (baseAttr.get(k) instanceof PerspectiveTransform) {
                    attributes.put(k, (Serializable) ((PerspectiveTransform) baseAttr.get(k)).clone());
                }

            }
        }
        return attributes;
    }

    /***/
    private void resetAttributes() {
        composited = (Boolean) attributes.get("composited");
        currentPreTransforming = (AffineTransform) attributes.get("currentPreTransforming");
        currentPreMirroring = (AffineTransform) attributes.get("currentPreMirroring");
        currentPreScaling = (AffineTransform) attributes.get("currentPreScaling");
        currentPrePerspective = (PerspectiveTransform) attributes.get("currentPrePerspective");
    }

    /**
     * Instances a volatile image as the data of this sprite.
     *
     * @param filename path to image (relative to CWD)
     * @param size dimension of the sprite
     * @param rsrcMode innerResource mode en/disabled (inn/outer .jar filepath)
     * @param mime image mime type (e.g. image/png)
     * @param buf whether to use buffered images or not
     * @discussion (comprehensive description)
     */
    public Sprite(String filename, boolean rsrcMode, String format, Dimension size, boolean buf) throws URISyntaxException {
        this(true, filename, rsrcMode, format, size, buf);
    }

    /**
     * Instances a volatile image as the data of this sprite.
     *
     * @param filename path to image (relative to CWD)
     * @param size dimension of the sprite
     * @param rsrcMode innerResource mode en/disabled (inn/outer .jar filepath)
     * @param mime image mime type (e.g. image/png)
     * @param buf whether to use buffered images or not
     * @discussion (comprehensive description)
     */
    public Sprite(boolean load, String filename, boolean rsrcMode, String format, Dimension size, boolean buf) throws URISyntaxException {
        super();
        loadDefaultAttributes(attributes);
        System.out.println("loading " + filename);
        URL url = getClass().getResource(filename);
        this.base = (rsrcMode) ? ((url == null) ? null : url.toURI()) : new File(filename).toURI();
        setThreadGroup(new CoalescedThreadsMonitor(getClass().getName() + "-TG " + base));
        this.load = load;
        setSize(size);
        innerResource = rsrcMode;
        diskCached = true;
        cache = (buf) ? BUFFERED : VOLATILE;
        this.size = size;
        src = filename;
        this.mt = new SfMediaTracker(this);
        this.mime = format;
        hash = hashCode();
    }

    /**
     * Instances a volatile image as the data of this sprite.
     *
     * @param stream input stream to image
     * @param mime image mime type
     * @param buf whether to set the Image BufferedImage or not (VolatileImage instead)
     * @param size dimension of the sprite
     * @see java.io.InputStream
     * @see java.awt.image.Image
     * @discussion (comprehensive description)
     */
    public Sprite(InputStream stream, String format, Dimension size, boolean buf) throws URISyntaxException {
        this(true, stream, format, size, buf);
    }

    /**
     * Instances a volatile image as the data of this sprite.
     *
     * @param stream input stream to image
     * @param mime image mime type
     * @param buf whether to set the Image BufferedImage or not (VolatileImage instead)
     * @param size dimension of the sprite
     * @see java.io.InputStream
     * @see java.awt.image.Image
     * @discussion (comprehensive description)
     */
    public Sprite(boolean load, InputStream stream, String format, Dimension size, boolean buf) throws URISyntaxException {
        super();
        loadDefaultAttributes(attributes);
        setThreadGroup(new CoalescedThreadsMonitor(getClass().getName() + "-TG " + base));
        this.load = load;
        setSize(size);
        System.out.println("loading " + stream);
        innerResource = false;
        diskCached = false;
        cache = (buf) ? BUFFERED : VOLATILE;
        this.size = size;
        src = stream;
        this.mt = new SfMediaTracker(this);
        this.mime = format;
        hash = hashCode();
    }

    /**
     * Sets image data of this sprite.
     *
     * @param data VolatileImage
     * @param size dimension of the sprite
     * @param mime image mime type
     * @discussion (comprehensive description)
     */
    public Sprite(VolatileImage data, String format, Dimension size) {
        this(true, data, format, size);
    }

    /**
     * Sets image data of this sprite.
     *
     * @param data VolatileImage
     * @param size dimension of the sprite
     * @param mime image mime type
     * @discussion (comprehensive description)
     */
    public Sprite(boolean load, VolatileImage data, String format, Dimension size) {
        super();
        loadDefaultAttributes(attributes);
        setThreadGroup(new CoalescedThreadsMonitor(getClass().getName() + "-TG " + base));
        this.load = load;
        setSize(size);
        diskCached = false;
        this.data = data;
        this.size = size;
        cache = VOLATILE;
        this.mt = new SfMediaTracker(this);
        this.mime = format;
        this.src = data.getSource();
        hash = hashCode();
    }

    /**
     * Sets image data of this sprite.
     *
     * @param data VolatileImage
     * @param size dimension of the sprite
     * @param mime image mime type
     * @discussion (comprehensive description)
     */
    public Sprite(BufferedImage data, String format, Dimension size) {
        this(true, data, format, size);
    }

    /**
     * Sets image data of this sprite.
     *
     * @param data VolatileImage
     * @param size dimension of the sprite
     * @param mime image mime type
     * @discussion (comprehensive description)
     */
    public Sprite(boolean load, BufferedImage data, String format, Dimension size) {
        super();
        loadDefaultAttributes(attributes);
        setThreadGroup(new CoalescedThreadsMonitor(getClass().getName() + "-TG " + base));
        this.load = load;
        setSize(size);
        diskCached = false;
        this.data = data;
        this.size = size;
        cache = BUFFERED;
        this.mt = new SfMediaTracker(this);
        this.mime = format;
        this.src = data.getSource();
        hash = hashCode();
    }

    /***/
    public int hashCode() {
        return size.hashCode() + ((base != null) ? base.hashCode() : (src != null) ? src.hashCode() : 0);
    }

    /***/
    public boolean equals(Object o) {
        if (o instanceof Sprite) {
            return ((Sprite) o).size.width == size.width && ((Sprite) o).size.height == size.height && (base != null && (((Sprite) o).base != null ? ((Sprite) o).base.equals(base) : src.equals(((Sprite) o).src)));
        } else {
            return false;
        }
    }

    /**
     * Refreshes size of sprite to current image data dimension.
     * @see #size
     * @see #data
     * @discussion (comprehensive description)
     */
    public void fitSizeToImage() throws InterruptedException {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            waitFor(trackerPty);
            if (data instanceof Image) {
                size = new Dimension(data.getWidth(this), data.getHeight(this));
            } else {
                size = new Dimension(getWidth(), getHeight());
            }
            monitor.notifyOnMonitor();
        }
        setPreferredSize(size);
        setSize(size);
    //System.out.println(src + ": " + size.width + "x" + size.height);
    }

    /**
     * get reading from
     *
     * @param n reader index default should be 0
     * @return ImageReader instance
     * @see #mime
     */
    private ImageReader getReader(int n) {
        ImageReader r = null;
        IIOReadProgressAdapter rpl = new IIOReadProgressAdapter(true);
        IIOReadWarningAdapter rwl = new IIOReadWarningAdapter(true);
        int i = 0;
        for (Iterator readers = ImageIO.getImageReadersByMIMEType(mime); readers.hasNext(); i++) {
            r = (ImageReader) readers.next();
            if (i != n) {
                continue;
            }
            r.addIIOReadProgressListener(rpl);
            synchronized (rProgList) {
                for (Iterator<IIOReadProgressListener> it = rProgList.iterator(); it.hasNext();) {
                    r.addIIOReadProgressListener(it.next());
                }
            }
            r.addIIOReadWarningListener(rwl);
            synchronized (rWarnList) {
                for (Iterator<IIOReadWarningListener> it = rWarnList.iterator(); it.hasNext();) {
                    r.addIIOReadWarningListener(it.next());
                }
            }
            if (i == n) {
                return r;
            }
            i++;
        }
        return null;
    }

    /**
     * get writing from
     *
     * @param n writer index default should be 0
     * @return ImageWriter instance
     * @see #mime
     */
    private ImageWriter getWriter(int n) {
        ImageWriter w = null;
        IIOWriteProgressAdapter wpl = new IIOWriteProgressAdapter(true);
        IIOWriteWarningAdapter wwl = new IIOWriteWarningAdapter(true);
        int i = 0;
        for (Iterator writers = ImageIO.getImageWritersByMIMEType(mime); writers.hasNext();) {
            w = (ImageWriter) writers.next();
            if (i != n) {
                continue;
            }
            w.addIIOWriteProgressListener(wpl);
            synchronized (wProgList) {
                for (Iterator<IIOWriteProgressListener> it = wProgList.iterator(); it.hasNext();) {
                    w.addIIOWriteProgressListener(it.next());
                }
            }
            w.addIIOWriteWarningListener(wwl);
            synchronized (wWarnList) {
                for (Iterator<IIOWriteWarningListener> it = wWarnList.iterator(); it.hasNext();) {
                    w.addIIOWriteWarningListener(it.next());
                }
            }
            if (i == n) {
                return w;
            }
            i++;
        }
        return null;
    }
    /***/
    private final String NO_SOURCE = "No Serializable Source Available";

    /** loads Image from known
     * @see #src*/
    public Object loadResource() {
        int currentPty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        boolean interrupt_ = false;
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (data instanceof VolatileImage) {
                while (((VolatileImage) data).contentsLost()) {
                    System.out.println("Sprite has lost contents!" + src);
                    data = (((VolatileImage) data).validate(GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration()) != VolatileImage.IMAGE_INCOMPATIBLE) ? data : null;
                    if (data == null) {
                        break;
                    }
                }
            }
            if (!(data instanceof Image) && load) {
                try {
                    invalidate();
                    ImageReader r = null;
                    ImageInputStream iis = null;
                    RandomAccessFile raf = null;
                    r = getReader(0);
                    if (!(src instanceof ImageProducer)) {
                        if (!innerResource && (src instanceof String)) {
                            File inputFile = new File((String) src);
                            if (!inputFile.isFile()) {
                                return data;
                            }
                            raf = new RandomAccessFile(inputFile, "r");
                            //raf.getChannel().lock(0L, Long.MAX_VALUE, true);
                            r.setInput(iis = new FileChannelImageInputStream(raf.getChannel()));
                        } else if (src instanceof InputStream) {
                            iis = ImageIO.createImageInputStream((InputStream) src);
                            iis.mark();
                            r.setInput(iis);
                        } else if (src instanceof ImageInputStream) {
                            iis = (ImageInputStream) src;
                            iis.reset();
                            r.setInput(iis);
                        } else if (src instanceof String) {
                            if (!src.equals(NO_SOURCE)) {
                                r.setInput(iis = ImageIO.createImageInputStream(getClass().getResourceAsStream((String) src)));
                            } else {
                                return data;
                            }
                        } else if (src instanceof File) {
                            raf = new RandomAccessFile((File) src, "r");
                            //raf.getChannel().lock(0L, Long.MAX_VALUE, true);
                            r.setInput(iis = new FileChannelImageInputStream(raf.getChannel()));
                        } else {
                            throw new IOException(new javax.media.Format(mime, src.getClass()).toString());
                        }
                        if (r.getInput() != null) {
                            data = (Image) r.readAsRenderedImage(r.getMinIndex(), r.getDefaultReadParam());
                        } else {
                            System.err.println("WARNING ! No suitable input image stream to read ! " + base);
                        }
                    } else {
                        data = createImage((ImageProducer) src);
                    }
                    if (data instanceof Image) {
                        trackImage(trackerPty, data, size);
                    }
                    if (iis != null) {
                        iis.close();
                    }
                    if (raf != null) {
                        raf.close();
                    }
                    if (r != null) {
                        r.dispose();
                    }
                } catch (IOException ex) {
                    ex.printStackTrace();
                }
            }
            monitor.notifyOnMonitor();
            if (interrupt_) {
                Thread.currentThread().interrupt();
                return null;
            }
            Thread.currentThread().setPriority(currentPty);
            return data;
        }
    }

    /***/
    private void waitFor(int id) throws InterruptedException {
        if (load) {
            System.out.println(base + " is waitin' for data " + id + " on MediaTracker " + mt.hashCode() + "...");
            while (!mt.checkID(id)) {
                System.out.print(".");
                mt.waitForID(id, 10);
            }
            if (!mt.isErrorID(id)) {
                System.out.println(base + " has loaded up " + id + ".");
            } else {
                System.err.println(base + " has error(s) " + id + ".");
                clearResource();
            }
        }
    }

    /** resizes image data to given size .
     * @param size dimension to resize to
     *@param obs observer to use
     * @return new resized image data
     * TODO transform matrixes management*/
    public Image resizeData(Dimension size, PerspectiveTransform ptx, AffineTransform tx, AffineTransform mirroring, AffineTransform scaling, Component pobs) throws InterruptedException, NullPointerException {
        obs = pobs;
        int currentPty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        AffineTransform baseTx = new AffineTransform();
        PerspectiveTransform basePtx = new PerspectiveTransform();
        loadResource();
        try {
            waitFor(trackerPty);
            final CoalescedThreadsMonitor monitor = imageSynch;
            synchronized (monitor.getMonitor(false)) {
                if (data == null) {
                    loadResource();
                    waitFor(trackerPty);
                    if (data == null) {
                        throw new NullPointerException("Sprite " + base + " had no data to draw ");
                    }
                }
                if (ptx == null) {
                    ptx = new PerspectiveTransform();
                }
                if (tx == null) {
                    tx = new AffineTransform();
                }
                if (mirroring == null) {
                    mirroring = new AffineTransform();
                }
                if (scaling == null) {
                    scaling = new AffineTransform();
                }
                // compute actual size with transforming
                baseTx.scale((double) ((float) size.width) / ((float) data.getWidth(obs)), (double) ((float) size.height) / ((float) data.getHeight(obs)));
                if (!ptx.equals(currentPrePerspective) || !tx.equals(currentPreTransforming) || !mirroring.equals(currentPreMirroring) || !scaling.equals(currentPreScaling) || size.width != data.getWidth(obs) || size.height != data.getHeight(obs)) {
                    invalidate();
                    Image ndata = (cache == BUFFERED) ? createBufferedImage(size, _type) : (Image) createVolatileImage(size);
                    trackImage(trackerPty, data, size);
                    waitFor(trackerPty);
                    if (data == null) {
                        throw new NullPointerException("Sprite " + base + " had no data to draw ");
                    }
                    // add transform (watch out to the order !)                                      
                    if (!ptx.equals(currentPrePerspective)) {
                        basePtx.concatenate(ptx);
                        currentPrePerspective = ptx;
                    }
                    if (!scaling.equals(currentPreScaling)) {
                        baseTx.preConcatenate(scaling);
                        currentPreScaling = scaling;
                    }                                      
                    if (!mirroring.equals(currentPreMirroring)) {
                        baseTx.preConcatenate(mirroring);
                        currentPreMirroring = mirroring;
                    }
                    if (!tx.equals(currentPreTransforming)) {
                        baseTx.preConcatenate(tx);
                        currentPreTransforming = tx;
                    }  
                    Rectangle2D transform = baseTx.createTransformedShape(new Rectangle(data.getWidth(obs), data.getHeight(obs))).getBounds2D();
                    baseTx.preConcatenate(AffineTransform.getScaleInstance((double) ((float) size.width) / ((float) transform.getWidth()), (double) ((float) size.height) / ((float) transform.getHeight())));
                    GraphicsJAI g = createGraphicsJAI(ndata.getGraphics(), obs);
                    if (basePtx.isIdentity()) {
                        System.out.println("Sprite " + base + " resizes to " + baseTx.getScaleX() + " * " + data.getWidth(obs) + " x " + baseTx.getScaleY() + " * " + data.getHeight(obs) + ".");
                        if (data instanceof RenderedImage) {
                            g.drawRenderedImage((RenderedImage) data, baseTx);
                        } else {
                            g.drawImage(data, baseTx, obs);
                        }
                        System.out.println("Sprite " + base + " transformed with " + baseTx);
                    } else {
                        basePtx.concatenate(baseTx);
                        ParameterBlock pb = new ParameterBlock();
                        pb.addSource(data);
                        pb.add(new WarpPerspective(basePtx));
                        System.out.println("Sprite " + base + " warped to : " + basePtx);
                        RenderedOp warpOp = JAI.create("warp", pb);
                        RenderedImage warped = (RenderedImage) warpOp;
                        g.drawRenderedImage(warped, new AffineTransform());
                    }
                    trackImage(trackerPty, ndata);
                    mt.removeImage(data);
                    monitor.notifyOnMonitor();
                    Thread.currentThread().setPriority(currentPty);
                    return ndata;
                } else {
                    monitor.notifyOnMonitor();
                    Thread.currentThread().setPriority(currentPty);
                    return data;
                }
            }
        } catch (Exception e) {
            if (e instanceof InterruptedException) {
                throw (InterruptedException) e;
            }
            if (e instanceof NullPointerException) {
                throw (NullPointerException) e;
            }
            e.printStackTrace();
            Thread.currentThread().setPriority(currentPty);
            return data;
        }
    }
    /***/
    protected boolean storeImage = true;

    /***/
    protected void setStoreImageEnabled(boolean b) {
        storeImage = b;
    }

    /***/
    protected boolean isStoreImageEnabled() {
        return storeImage;
    }

    /** writes Sprite instance to given output
     * @param out OOutputStream to write to*/
    private void writeObject(ObjectOutputStream out) throws IOException {
        int currentPty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        validate();
        if (load) {
            try {
                final CoalescedThreadsMonitor monitor = validateMonitor;
                synchronized (monitor.getMonitor(false)) {
                    while (validating || !valid) {
                        monitor.waitOnMonitor(10);
                    }
                    storeCurrentAttributes();
                    out.defaultWriteObject();
                    final CoalescedThreadsMonitor monitor1 = imageSynch;
                    synchronized (monitor1.getMonitor(false)) {
                        if (data instanceof Image) {
                            out.writeBoolean(true);
                            if (!(src instanceof File)) {
                                File dir = new File(image_dir);
                                dir.mkdirs();
                                File newSrc = File.createTempFile("sp3_", "" + hash, dir);
                                ((File) newSrc).deleteOnExit();
                                RandomAccessFile raf = new RandomAccessFile((File) newSrc, "rws");
                                writeImageToFile(raf);
                                raf.close();
                                src = newSrc;
                            }
                            out.writeObject(src);
                            out.writeLong(((File) src).length());
                            if (storeImage) {
                                writeImageToIOStream(ImageIO.createImageOutputStream(out));
                            }
                        } else {
                            out.writeBoolean(false);
                            if (src instanceof Serializable) {
                                out.writeObject(src);
                            } else {
                                out.writeObject(NO_SOURCE);
                            }
                        }
                        monitor1.notifyOnMonitor();
                    }
                    monitor.notifyOnMonitor();
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
                Thread.currentThread().interrupt();
                return;
            }
        } else {
            storeCurrentAttributes();
            out.defaultWriteObject();
            if (src instanceof Serializable) {
                out.writeObject(src);
            } else {
                out.writeObject(NO_SOURCE);
            }
        }
        Thread.currentThread().setPriority(currentPty);
    }

    /** reads Sprite instanced from given input
     * @param in OInputStream to read from*/
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        int currentPty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        in.defaultReadObject();
        resetAttributes();
        valid = false;
        setThreadGroup(new CoalescedThreadsMonitor(getClass().getName() + "-TG " + base));
        refQueue = new ReferenceQueue<Image>();
        mt = new SfMediaTracker(obs = this);
        rProgList = Collections.synchronizedSet(new HashSet<IIOReadProgressListener>());
        wProgList = Collections.synchronizedSet(new HashSet<IIOWriteProgressListener>());
        rWarnList = Collections.synchronizedSet(new HashSet<IIOReadWarningListener>());
        wWarnList = Collections.synchronizedSet(new HashSet<IIOWriteWarningListener>());
        if (load) {
            long len = 0;
            boolean image = false;
            boolean createTemp = true;
            final CoalescedThreadsMonitor monitor = imageSynch;
            synchronized (monitor.getMonitor(false)) {
                if (image = in.readBoolean()) {
                    try {
                        src = in.readObject();
                        len = in.readLong();
                        if (src.equals(NO_SOURCE)) {
                            createTemp = false;
                        } else if (src instanceof File) {
                            innerResource = false;
                            if (((File) src).exists() && ((File) src).length() == len) {
                                createTemp = false;
                            }
                        }
                        if (storeImage) {
                            if (createTemp) {
                                ImageReader r = getReader(0);
                                ImageInputStream is = ImageIO.createImageInputStream(in);
                                if (is instanceof ImageInputStream) {
                                    r.setInput(is);
                                    data = (Image) r.readAsRenderedImage(r.getMinIndex(), r.getDefaultReadParam());
                                    trackImage(trackerPty, data);
                                } else {
                                    System.err.println("WARNING ! No suitable image input stream to read !" + base);
                                }
                            } else {
                                in.skipBytes((int) len);
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                } else {
                    src = in.readObject();
                }
                monitor.notifyOnMonitor();
                if (image) { // store image if necessary
                    if (storeImage) {
                        if (createTemp) {
                            File dir = new File(image_dir);
                            dir.mkdirs();
                            File newSrc = File.createTempFile("sp3_", "" + hash, dir);
                            ((File) newSrc).deleteOnExit();
                            RandomAccessFile raf = new RandomAccessFile((File) newSrc, "rws");
                            try {
                                writeImageToFile(raf);
                                innerResource = false;
                                src = newSrc;
                            } catch (InterruptedException e) {
                                e.printStackTrace();
                                Thread.currentThread().interrupt();
                            } finally {
                                raf.close();
                            }
                        }
                    }
                }
            }
        } else {
            src = in.readObject();
        }
        invalidate();
        Thread.currentThread().setPriority(currentPty);
    }

    /***/
    public void writeImageToFile(RandomAccessFile raf) throws IOException, InterruptedException {
        ImageOutputStream ios;
        //FileLock fl = raf.getChannel().lock();
        raf.getChannel().truncate(0);
        writeImageToIOStream(ios = new FileChannelImageOutputStream(raf.getChannel()));
    //fl.release();
    }

    /***/
    public void writeImageToIOStream(ImageOutputStream output) throws IOException, InterruptedException {
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        System.out.println("Writing sprite image to stream...");
        if (output == null) {
            System.err.println("WARNING ! No suitable image output stream to write ! " + base);
            return;
        }
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (!(data instanceof BufferedImage)) {
                data = convertToBuffered(data, this);
            }
            if (!(data instanceof RenderedImage)) {
                throw new IOException("READ/WRITE ERROR : Sprite image isn't compatible with RenderedImage. " + data.getClass().getName());
            }
            ImageWriter w = getWriter(0);
            IIOMetadata meta = w.getDefaultImageMetadata(ImageTypeSpecifier.createFromRenderedImage((RenderedImage) data), w.getDefaultWriteParam());
            w.setOutput(output);
            waitFor(trackerPty);
            if (data == null) {
                throw new NullPointerException("Sprite " + base + " had no data to draw ");
            }
            IIOImage iioimage = new IIOImage((RenderedImage) data, null, meta);
            if (w.canInsertImage(-1)) {
                w.writeInsert(-1, iioimage, w.getDefaultWriteParam());
            } else {
                w.write(meta, iioimage, w.getDefaultWriteParam());
            }
            if (w != null) {
                w.dispose();
            }
            System.out.println("writing is done.");
            monitor.notifyOnMonitor();
            Thread.currentThread().setPriority(pty);
        }
    }

    /***/
    public BufferedImage convertToBuffered(Image data, Component pobs) throws InterruptedException, NullPointerException {
        this.obs = pobs;
        Dimension d;
        BufferedImage bdata = createBufferedImage(d = new Dimension(data.getWidth(obs), data.getHeight(obs)), _type);
        final CoalescedThreadsMonitor monitor1 = imageSynch;
        synchronized (monitor1.getMonitor(false)) {
            waitFor(trackerPty);
            if (data == null) {
                throw new NullPointerException("Sprite " + base + " had no data to draw ");
            }
            if (data instanceof RenderedImage) {
                createGraphicsJAI(bdata.getGraphics(), obs).drawRenderedImage((RenderedImage) data, new AffineTransform());
            } else {
                createGraphicsJAI(bdata.getGraphics(), obs).drawImage(data, new AffineTransform(), obs);
            }
            trackImage(trackerPty, bdata, size);
            mt.removeImage(data);
            monitor1.notifyOnMonitor();
        }
        return bdata;
    }

    /***/
    public VolatileImage convertToVolatile(Image data, Component pobs) throws InterruptedException, NullPointerException {
        obs = pobs;
        Dimension d;
        VolatileImage vdata = createVolatileImage(d = new Dimension(data.getWidth(obs), data.getHeight(obs)));
        final CoalescedThreadsMonitor monitor1 = imageSynch;
        synchronized (monitor1.getMonitor(false)) {
            waitFor(trackerPty);
            if (data == null) {
                throw new NullPointerException("Sprite " + base + " had no data to draw ");
            }
            if (data instanceof RenderedImage) {
                createGraphicsJAI(vdata.getGraphics(), obs).drawRenderedImage((RenderedImage) data, new AffineTransform());
            } else {
                createGraphicsJAI(vdata.getGraphics(), obs).drawImage(data, new AffineTransform(), obs);
            }
            trackImage(trackerPty, vdata, size);
            mt.removeImage(data);
            monitor1.notifyOnMonitor();
        }
        return vdata;
    }

    /** sets observer and tracker used by the Sprite. Note that a new MediaTracker will be instanciated. if you want to focalize on an unique tracker for many sprite don't use this method but use setMt() instead.
     * @see Graphics
     * @see MediaTracker
     * @param obs component that will track and observe (care with observer that the image might be reloaded again)
     * @see #setMT(MediaTracker)
     * public void setObserver(Component obs) {
     * this.obs = obs;
     * clearImage();
     * mt = new MediaTracker(obs);
     * }*/
    /** returns base URI to this sprite if it references a file
     * @see #src
     * @return link to this sprite or null if src isn't a file reference */
    public URI getBase() {
        return base;
    }

    /** repaints the component with the Sprite data. used by Swing EDT when called
     * @see JComponent#repaint()
     * @see JComponent#update(Graphics)
     * @param g Graphics instance*/
    protected void paintComponent(Graphics g) {
        try {
            validate();
            //pack();
            draw(this, (Graphics2D) g);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    /***/
    public void update(Graphics g) {
        if (isOpaque()) {
            g.clearRect(0, 0, getWidth(), getHeight());
        }
        paint(g);
    }

    /***/
    public boolean isOpaque() {
        return cache == VOLATILE;
    }

    /***/
    public void setOpaque(boolean b) {
        /* final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {*/
        if (isOpaque() != b) {
            cache = (b) ? VOLATILE : BUFFERED;
            invalidate();
        } else {
            cache = (b) ? VOLATILE : BUFFERED;
        }
    /*  monitor.notifyOnMonitor();
    }*/
    }
    /***/
    private transient Thread pack = null;

    /** used for packing the sprite data and set it ready to paint */
    public void pack() {
        if (!valid) {
            validate();
        }
        if (load) {
            if (pack instanceof Thread) {
                if (pack.isAlive()) {
                    return;
                }
            }
            Thread t = pack = new Thread(packMonitor, new Runnable() {

                public void run() {
                    boolean interrupt_ = false;
                    final CoalescedThreadsMonitor monitor = validateMonitor;
                    try {
                        synchronized (monitor.getMonitor(false)) {
                            while ((!valid || validating) && load) {
                                monitor.waitOnMonitor(10);
                            }
                            packing = true;
                            final CoalescedThreadsMonitor monitor0 = imageSynch;
                            synchronized (monitor0.getMonitor(false)) {
                                try {
                                    waitFor(trackerPty);
                                    if (data == null) {
                                        throw new NullPointerException("Sprite " + base + " had no data to draw ");
                                    }
                                } catch (NullPointerException ex) {
                                    ex.printStackTrace();
                                } finally {
                                    monitor0.notifyOnMonitor();
                                    monitor.notifyOnMonitor();
                                }
                            }
                        }
                    } catch (InterruptedException ex) {
                        ex.printStackTrace();
                        interrupt_ = true;
                    } finally {
                        final CoalescedThreadsMonitor monitor1 = packMonitor;
                        synchronized (monitor1.getMonitor(false)) {
                            packing = false;
                            monitor1.notifyAllOnMonitor();
                        }
                        if (interrupt_) {
                            Thread.currentThread().interrupt();
                            return;
                        }
                    }
                }
            });
            t.setPriority(Thread.MAX_PRIORITY);
            t.start();
        }
    }

    /** draws the current data on component
     * @see #draw(Graphics2D, AffineTransform)
     * @param g2 Graphics2D instance to draw to*/
    public boolean draw(Component obs, Graphics2D g2) throws InterruptedException {
        return draw(obs, g2, null, null);
    }

    /***/
    public static HashMap<RenderingHints.Key, Object> _getRenderingHints() {
        HashMap<RenderingHints.Key, Object> render = new HashMap<RenderingHints.Key, Object>();
        render.put(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
        render.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);
        render.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
        render.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
        render.put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
        return render;
    }

    /** draws with Transform concatenated.
     * @param g Graphics2D instance to draw to
     * @param tx transform to be applied to drawing (unstable with some image types)*/
    public boolean draw(Component pobs, Graphics2D g, AffineTransform tx, PerspectiveTransform ptx) throws InterruptedException {
        obs = pobs;
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(paintMonitor.getMaxPriority());
        boolean interrupt_ = false;
        boolean complete = false;
        final CoalescedThreadsMonitor monitor = validateMonitor;
        synchronized (monitor.getMonitor(false)) {
            if (validating) {
                System.err.println("validating " + base + " sprite's waitin'...");
            }
            while (validating || !valid) {
                System.err.print(".");
                monitor.waitOnMonitor(10);
            }
            painting = true;
            final CoalescedThreadsMonitor monitor1 = imageSynch;
            synchronized (monitor1.getMonitor(false)) {
                try {
                    waitFor(trackerPty);
                    if (data == null) {
                        throw new NullPointerException("Sprite " + base + " has no data to draw ! try (re)validate().");
                    } else {
                        System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>>>>draw " + base + " image size=" + data.getWidth(obs) + "x" + data.getHeight(obs) + " on clip size=" + size);
                    }
                    System.err.print("OK\n\r");
                    AffineTransform baseTX = new AffineTransform();
                    if (tx != null) {
                        baseTX.concatenate(tx);
                    }
                    PerspectiveTransform basePtx = new PerspectiveTransform();
                    if (ptx != null) {
                        basePtx.concatenate(ptx);
                    }
                    GraphicsJAI g2 = createGraphicsJAI(g, obs);
                    g2.setRenderingHints(_getRenderingHints());
                    g2.drawString("sprite", 0, 0);
                    if (data instanceof Image) {
                        g2.drawString("sprite ok", 0, 0);
                        if (baseTX.getScaleX() == 0.0 || baseTX.getScaleY() == 0.0 || baseTX.getDeterminant() == 0.0) {
                            System.out.println("transform " + base + " contains Zero, so is avoided. TX=" + baseTX);
                            throw new InterruptedException(base + " transform failed! " + baseTX);
                        }
                        if (data.getWidth(obs) <= 0 || data.getHeight(obs) <= 0) {
                            throw new InterruptedException(base + " image data size was not available!");
                        }
                        if (!basePtx.isIdentity()) {
                            basePtx.concatenate(baseTX);
                        }
                        if (!baseTX.isIdentity() && basePtx.isIdentity()) {
                            Rectangle transform = baseTX.createTransformedShape(new Rectangle(data.getWidth(obs), data.getHeight(obs))).getBounds();
                            trackImage(trackerPty, data, transform.getSize());
                            System.out.println("TXShape : " + transform);
                            if (transform.getSize().height <= 1 || transform.getSize().width <= 1) {
                                throw new InterruptedException(base + " transform failed! " + transform);
                            }
                        }
                        waitFor(trackerPty);
                        if (data == null) {
                            throw new NullPointerException("Sprite " + base + " had no data to draw ");
                        } else {
                            if (basePtx.isIdentity()) {
                                if (data instanceof RenderedImage) {
                                    g2.drawRenderedImage((RenderedImage) data, baseTX);
                                    complete = true;
                                } else {
                                    complete = g2.drawImage(data, baseTX, obs);
                                }
                            } else {
                                ParameterBlock pb = new ParameterBlock();
                                pb.addSource(data);
                                pb.add(new WarpPerspective(basePtx));
                                System.out.println("Sprite " + base + " warped to : " + ptx);
                                RenderedOp warpOp = JAI.create("warp", pb);
                                RenderedImage warped = (RenderedImage) warpOp;
                                g.drawRenderedImage(warped, new AffineTransform());
                            }
                        }
                    } else {
                        throw new NullPointerException("Sprite " + base + " had no data to draw ");
                    }
                } catch (InterruptedException e) {
                    e.printStackTrace();
                    interrupt_ = true;
                } catch (NullPointerException e) {
                    e.printStackTrace();
                } finally {
                    monitor1.notifyOnMonitor();
                    monitor.notifyOnMonitor();
                }
            }
        }
        final CoalescedThreadsMonitor monitor1 = paintMonitor;
        synchronized (monitor1.getMonitor(false)) {
            painting = false;
            monitor1.notifyAllOnMonitor();
        }
        if (interrupt_) {
            throw new InterruptedException("Sprite " + base + " caught an interruption.");
        }
        markFPS();
        Thread.currentThread().setPriority(pty);
        return complete;
    }

    /** tells whether the Sprite has a disk cached data available
     * @return true or false*/
    public boolean isDiskCached() {
        return diskCached;
    }

    /**
     *
     * public void setRendering(RenderingHints rendering) {
     * this.rendering = rendering;
     * } */
    /** Tells whether buffer is enabled for this Sprite
     * @see #cache
     * @return true if buffer is activated, false otherwise
     */
    public boolean isBuffered() {
        return (cache == BUFFERED) ? true : false;
    }

    /** convert to BufferedImage
     * @return buffered image*/
    public BufferedImage toBuffered() throws InterruptedException {
        validate();
        //pack();
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            monitor.notifyOnMonitor();
            return convertToBuffered(data, this);
        }
    }

    /** convert to VolatileImage
     * @return volatile image*/
    public VolatileImage toVolatile() throws InterruptedException {
        validate();
        //pack();
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            monitor.notifyOnMonitor();
            return convertToVolatile(data, this);
        }
    }

    /**
     * Instances a new BufferedImage from an image object.
     * @discussion (comprehensive description)
     * @param size dimensions
     * @return BufferedImage created
     * @see java.awt.image.BufferedImage
     */
    public static BufferedImage createBufferedImage(Dimension size, int type) {
        BufferedImage bdata = new BufferedImage(size.width, size.height, type);
        return bdata;
    }

    /**
     * Instances a new VolatileImage from an image object.
     * @discussion (comprehensive description)
     * @param size dimensions
     * @return VolatileImage created
     * @see java.awt.image.VolatileImage
     */
    public static VolatileImage createVolatileImage(Dimension size) {
        VolatileImage vdata = GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration().createCompatibleVolatileImage(size.width, size.height);
        return vdata;
    }

    /**
     * Returns width of sprite.
     * @discussion (comprehensive description)
     * @return width in pixels
     * @see #resizeData(Dimension, Component)
     */
    public int getWidth() {
        int w = size.width;
        return w;
    }

    /**
     * Returns height of sprite.
     * @discussion (comprehensive description)
     * @return height in pixels
     * @see #resizeData(Dimension, Component)
     */
    public int getHeight() {
        int h = size.height;
        return h;
    }

    /**
     * Returns data field. Image instance.
     * @discussion (comprehensive description)
     * @return sprite data
     */
    public Image getImage(Component obs) throws InterruptedException {
        System.out.println(src + ": fetching sprite data...");
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(packMonitor.getMaxPriority());
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (spm instanceof SpritesCacheManager) {
                try {
                    data = (Image) spm.memorySensitiveCallback("resizeData", this, new Object[]{size, perspective, transforming, mirroring, scaling, obs}, new Class[]{Dimension.class, PerspectiveTransform.class, AffineTransform.class, AffineTransform.class, AffineTransform.class, Component.class});
                } catch (Throwable ex) {
                    ex.printStackTrace();
                    throw new InterruptedException();
                }
            } else {
                data = resizeData(size, perspective, transforming, mirroring, scaling, obs);
            }
            Thread.currentThread().setPriority(pty);
            monitor.notifyOnMonitor();
            return data;
        }
    }

    /***/
    public boolean isCompositeEnabled() {
        return compositeEnabled;
    }

    /***/
    public void setCompositeEnabled(boolean b) {
        /*final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {*/
        if (compositeEnabled != b) {
            compositeEnabled = b;
            invalidate();
        } else {
            compositeEnabled = b;
        }
    /*     monitor.notifyOnMonitor();
    }*/
    }

    /***/
    public Composite getComposite() {
        return cps;
    }

    public boolean isComposited() {
        return composited;
    }

    /***/
    public void setComposited(boolean b) {
        composited = b;
    }

    /***/
    public void setComposite(Composite cps) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        if (this.cps instanceof Composite) {
            if (!this.cps.equals(cps)) {
                this.cps = cps;
                invalidate();
            }
        } else if (cps instanceof Composite) {
            this.cps = cps;
            invalidate();
        } else {
            this.cps = cps;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /***/
    public Paint getPaint() {
        return pnt;
    }

    /***/
    public void setPaint(Paint pnt) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        if (this.pnt instanceof Paint) {
            if (!this.pnt.equals(pnt)) {
                this.pnt = pnt;
                invalidate();
            }
        } else if (pnt instanceof Paint) {
            this.pnt = pnt;
            invalidate();
        } else {
            this.pnt = pnt;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /***/
    public Color getColor() {
        return clr;
    }

    /***/
    public void setColor(Color clr) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        if (this.clr instanceof Color) {
            if (!this.clr.equals(clr)) {
                this.clr = clr;
                invalidate();
            }
        } else if (clr instanceof Color) {
            this.clr = clr;
            invalidate();
        } else {
            this.clr = clr;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /** returns the current Graphics. Clipping fits the current size, background color is set to Black and front color to Yellow.
     * @see #data
     * @return Graphics instance of JAI */
    public Graphics getGraphics() {
        Graphics g1 = super.getGraphics();
        if (size != null) {
            g1.setClip(0, 0, size.width, size.height);
        }
        GraphicsJAI g = createGraphicsJAI(g1, this);
        return g;
    }

    /** returns current size
     * @return dimension*/
    public Dimension getSize() {
        return size;
    }

    /**
     * Returns the image sprite scaled to the gicen factors.
     * @param zoomX width scale factor
     * @param zoomY height scale factor
     *@param obs ImageObserver where further drawing will be done
     * @return new instance of Image
     */
    public Image getScaled(double zoomX, double zoomY, Component obs) throws InterruptedException {
        System.out.println(src + ": scaling image X:" + zoomX + "x Y:" + zoomY + "x ...");
        Dimension newSize = new Dimension((int) Math.ceil(size.width * zoomX), (int) Math.ceil(size.height * zoomY));
        if (spm instanceof SpritesCacheManager) {
            try {
                return (Image) spm.memorySensitiveCallback("resizeData", this, new Object[]{size, perspective, transforming, mirroring, scaling, obs}, new Class[]{Dimension.class, PerspectiveTransform.class, AffineTransform.class, AffineTransform.class, AffineTransform.class, Component.class});
            } catch (Throwable ex) {
                ex.printStackTrace();
                throw new InterruptedException();
            }
        } else {
            return resizeData(size, perspective, transforming, mirroring, scaling, obs);
        }
    }

    /**
     * Zooms the sprite to the given factor. this method synchronizes on this sprite.
     * @discussion (comprehensive description)
     */
    private void zoom(Component obs) throws InterruptedException {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            data = getScaled(zoom, zoom, obs);
            monitor.notifyOnMonitor();
        }
        fitSizeToImage();
    }

    /** return the current zoom value
     * @return zoom value 1.0 is no zoom*/
    public double getZoomValue() {
        return zoom;
    }

    /***/
    public boolean isZoomEnabled() {
        return zoom != 1.0;
    }

    /** activates scale Transform
     * @param b en/disabled
     * @param zoom zoom value
     */
    public void setZoomEnabled(boolean b, double zoom) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        scaling = new AffineTransform();
        scaling.scale((b) ? zoom : 1.0, (b) ? zoom : 1.0);
        //setFlipEnabled(true, mirror);
        if (this.zoom != zoom || (!b && 1.0 != this.zoom)) {
            this.zoom = (b) ? zoom : 1.0;
            invalidate();
        } else {
            this.zoom = (b) ? zoom : 1.0;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /***/
    public boolean isFlipEnabled() {
        return mirror != NONE;
    }

    /** activates flip Transform (mirroring)
     * @param b en/disabled
     * @param mirror orientation of the mirror*/
    public void setFlipEnabled(boolean b, int mirror) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        mirroring = new AffineTransform();
        switch ((b) ? mirror : NONE) {
            case HORIZONTAL:
                mirroring.translate((double) Math.abs((float) scaling.getScaleX() * (float) size.width), 0);
                mirroring.scale(-1.0, 1.0);
                break;
            case VERTICAL:
                mirroring.translate(0, (double) Math.abs((float) scaling.getScaleY() * (float) size.height));
                mirroring.scale(1.0, -1.0);
                break;
            default:
                break;
        }
        if (this.mirror != mirror || (!b && this.mirror != NONE)) {
            this.mirror = (b) ? mirror : NONE;
            invalidate();
        } else {
            this.mirror = (b) ? mirror : NONE;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /**
     * Flips the current sprite.
     * This method synchronizes to sprite.
     */
    private void flip(Component obs) throws InterruptedException {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            data = getFlip(mirror, obs);
            monitor.notifyOnMonitor();
        }
        System.out.println(src + " flipped");
    }

    /** Returns a transformed image by the givsn orientation-symetrical-axis
     *@param orientation the mirror orientation to perform the flip-transform
     *@param obs the observer
     * @see ImageObserver
     *@return tranformed image
     */
    public Image getFlip(int orientation, Component pobs) throws InterruptedException, NullPointerException {
        obs = pobs;
        validate();
        //pack();
        Image bdata = (cache == VOLATILE) ? (Image) createVolatileImage(size) : createBufferedImage(size, _type);
        AffineTransform tx;
        switch (orientation) {
            case HORIZONTAL:
                tx = AffineTransform.getTranslateInstance(size.width, 0);
                tx.scale(-1.0, 1.0);
                break;
            case VERTICAL:
                tx = AffineTransform.getTranslateInstance(0, size.height);
                tx.scale(1.0, -1.0);
                break;
            default:
                tx = AffineTransform.getTranslateInstance(0, 0);
                break;
        }
        draw(obs, (Graphics2D) bdata.getGraphics(), tx, null);
        trackImage(trackerPty, bdata, size);
        return bdata;
    }

    /** returns mirror orientation
     * @see #mirror
     * @return mirror orientation*/
    public int getMirrorValue() {
        return mirror;
    }

    /** changes mirror orientation
     * @param orientation*/
    public void setMirror(int orientation) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        if (mirror != orientation) {
            mirror = orientation;
            invalidate();
        } else {
            mirror = orientation;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /** sets the size of this sprite
     * @param size dimension of the sprite*/
    public void setSize(Dimension size) {
        super.setSize(size);
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        if (this.size instanceof Dimension) {
            if (!this.size.equals(size)) {
                this.size = size;
                //setFlipEnabled(true, mirror);
                invalidate();
            }
        } else if (size instanceof Dimension) {
            this.size = size;
            //setFlipEnabled(true, mirror);
            invalidate();
        } else {
            this.size = size;
        //setFlipEnabled(true, mirror);
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /** sets the size of this sprite
     * @param width width dimension of the sprite
     * @param height dimension of the sprite*/
    public void setSize(int width, int height) {
        super.setSize(width, height);
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        Dimension d = new Dimension(width, height);
        if (this.size instanceof Dimension) {
            if (!this.size.equals(d)) {
                this.size = d;
                //setFlipEnabled(true, mirror);
                invalidate();
            }
        } else if (d instanceof Dimension) {
            this.size = d;
            //setFlipEnabled(true, mirror);
            invalidate();
        } else {
            this.size = d;
        //setFlipEnabled(true, mirror);
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /** sets the preferred size of this sprite
     * @param size preferred dimesnion of this sprite*/
    public void setPreferredSize(Dimension size) {
        super.setPreferredSize(size);
    }

    /**
     * resets Transform
     *
     * @see #transforming
     */
    public void resetTX() {
        setTX(new AffineTransform());
    }

    /***/
    public void resetPTX() {
        setPTX(new PerspectiveTransform());
    }

    /***/
    public void setPTX(PerspectiveTransform ptx) {
        if (this.perspective instanceof PerspectiveTransform) {
            if (!this.perspective.equals(ptx)) {
                this.perspective = ptx;
                invalidate();
            }
        } else if (ptx instanceof PerspectiveTransform) {
            this.perspective = ptx;
            invalidate();
        } else {
            this.perspective = ptx;
        }
    }

    /***/
    public PerspectiveTransform getPTX() {
        return perspective;
    }

    /**
     * set Transform
     *
     * @param transforming the transform to apply
     * @see #transforming
     */
    public void setTX(AffineTransform tx) {
//        final CoalescedThreadsMonitor monitor = imageSynch;
//        synchronized (monitor.getMonitor(false)) {
        if (this.transforming instanceof AffineTransform) {
            if (!this.transforming.equals(tx)) {
                this.transforming = tx;
                invalidate();
            }
        } else if (tx instanceof AffineTransform) {
            this.transforming = tx;
            invalidate();
        } else {
            this.transforming = tx;
        }
//            monitor.notifyOnMonitor();
//        }
    }

    /**
     * get Transform
     *
     * @return transform instance
     * @see #transforming
     */
    public AffineTransform getTX() {
        return transforming;
    }

    /** Returns source of image.
     * @see #src
     * @return object source of this sprite
     */
    public Object getSource() {
        return src;
    }

    /**
     * Tracks the image for loading with the current MediaTracker.
     * @discussion (comprehensive description)
     */
    private void trackImage(int id, Image data) {
        if (data != null) {
            mt.addImage(data, id);
            //data.setAccelerationPriority(1.0f);
            ref = new PhantomReference(data, refQueue);
        //statusImage();
        }
        cleanup();
    }

    /** cleanup referenceQueue */
    public void cleanup() {
        Reference ref;
        while ((ref = refQueue.poll()) instanceof Reference) {
            ref.clear();
        }
    }

    /** tracks the image for loading at specified dimensions
     * @see #waitFor()
     * @param size dimesnion at which the sprite will be drawn */
    private void trackImage(int id, Image data, Dimension size) {
        if (data != null) {
            mt.addImage(data, id, size.width, size.height);
            //data.setAccelerationPriority(1.0f);
            ref = new PhantomReference(data, refQueue);
        //statusImage();
        }
        cleanup();
    }

    /** changes referenceQueue (usually to get the same refQueue for a couple of Sprite in Animation)
     * @param refQueue the reference queue
     * public void setRefQueue(ReferenceQueue queue) {
     * refQueue = queue;
     * }
     */
    /** status image
     * @return current status taken from current
     * @see MediaTracker#statusID(int, boolean)
     *@see MediaTracker#LOADING
     * @see MediaTracker#ABORTED
     * @see MediaTracker#ERRORED
     * @see MediaTracker#COMPLETE*/
    public int statusImage() {
        return mt.statusID(trackerPty, true);
    }

    /**
     * Clears image tracking, and removes memory trace of it.
     * @see MediaTracker#removeImage(Image)
     * @discussion (comprehensive description)
     */
    public Object clearResource() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            resetAttributes();
            invalidate();
            if (data != null) {
                mt.removeImage(data);
            }
            data = null;
            monitor.notifyOnMonitor();
        }
        cleanup();
        if (imageSynch instanceof ThreadGroup) {
            imageSynch.interrupt();
        }
        /*        if (readMonitor instanceof ThreadGroup) {
        readMonitor.interrupt();
        }*/
        if (paintMonitor instanceof ThreadGroup) {
            paintMonitor.interrupt();
        }
        if (packMonitor instanceof ThreadGroup) {
            packMonitor.interrupt();
        }
        if (validateMonitor instanceof ThreadGroup) {
            validateMonitor.interrupt();
        }
        /*if (writeMonitor instanceof ThreadGroup) {
        writeMonitor.interrupt();
        }*/
        System.err.println("XXXXXX Clearing cached image " + src);
        return null;
    }

    /** clones the sprite in another identical sprite
     * @return the cloned Sprite*/
    public Object clone() {
        Sprite cloned = null;
        try {
            validate();
            cloned = (Sprite) super.clone();
            cloned.base = (base instanceof URI) ? new URI(base.toString()) : null;
            cloned.obs = obs;
            cloned.validating = false;
            cloned.packing = false;
            cloned.painting = false;
            cloned.setThreadGroup(getThreadGroup());
            cloned.rProgList = Collections.synchronizedSet(new HashSet<IIOReadProgressListener>());
            cloned.wProgList = Collections.synchronizedSet(new HashSet<IIOWriteProgressListener>());
            cloned.rWarnList = Collections.synchronizedSet(new HashSet<IIOReadWarningListener>());
            cloned.wWarnList = Collections.synchronizedSet(new HashSet<IIOWriteWarningListener>());
            cloned.attributes = cloneAttributes();
            cloned.perspective = (PerspectiveTransform) perspective.clone();
            cloned.currentPrePerspective = (PerspectiveTransform) currentPrePerspective.clone();
            cloned.transforming = (AffineTransform) transforming.clone();
            cloned.currentPreTransforming = (AffineTransform) currentPreTransforming.clone();
            cloned.mirroring = (AffineTransform) mirroring.clone();
            cloned.currentPreMirroring = (AffineTransform) currentPreMirroring.clone();
            cloned.scaling = (AffineTransform) scaling.clone();
            cloned.currentPreScaling = (AffineTransform) currentPreScaling.clone();
            cloned.src = src;
            cloned.setPreferredSize((Dimension) getPreferredSize().clone());
            cloned.size = (Dimension) size.clone();
            cloned.refQueue = new ReferenceQueue<Image>();
            cloned.mt = mt;
            cloned.data = (cache == BUFFERED) ? createBufferedImage(cloned.size, cloned._type) : (Image) createVolatileImage(size);
            GraphicsJAI g = cloned.createGraphicsJAI(cloned.data.getGraphics(), cloned);
            draw(cloned.obs, g);
            cloned.trackImage(cloned.trackerPty, cloned.data);
            cloned.invalidate();
            return cloned;
        } catch (InterruptedException ex) {
            ex.printStackTrace();
            Thread.currentThread().interrupt();
            return null;
        } catch (URISyntaxException e) {
            e.printStackTrace();
            return null;
        } catch (CloneNotSupportedException e) {
            e.printStackTrace();
            return null;
        }
    }

    /** Called by the GC when object is about to be destructed. it is overriden to immediately clear the resources used by this sprite*/
    public void finalize() throws Throwable {
        validateMonitor.cancelAll(true);
        paintMonitor.cancelAll(true);
        packMonitor.cancelAll(true);
        clearResource();
        super.finalize();
        System.err.println("GC's destroyed " + src);
    }

    /** Checks source or source file access
     * @return whether the source is OK or not*/
    public boolean checkSource() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            monitor.notifyOnMonitor();
            return (src instanceof String) ? new File((String) src).canRead() : ((src instanceof InputStream) ? true : (data != null));
        }
    }

    /**
     * en/disable innerResource mode
     *
     * @param b en/disabled
     * @see #innerResource
     */
    public void setInnerResourceEnabled(boolean b) {
        innerResource = b;
    }

    /**
     * checks whether innerResource mode is en/disabled
     *
     * @return en/disabled
     */
    public boolean isResource() {
        return innerResource;
    }

    /***/
    public String toString() {
        return src.toString();
    }
    /***/
    private transient boolean valid = false;

    /***/
    public void revalidate() {
        invalidate();
        validate();
    }

    /***/
    public void invalidate() {
        super.invalidate();
        valid = false;
    }

    /***/
    public boolean isValid() {
        return valid;
    }
    /***/
    private transient Thread validate = null;
    /***/
    private transient SpritesCacheManager spm;

    /***/
    public void setSPM(SpritesCacheManager spm) {
        this.spm = spm;
    }

    /***/
    public void validate() {
        super.validate();
        if (load) {
            if (validate instanceof Thread) {
                if (validate.isAlive()) {
                    return;
                }
            }
            Thread t = validate = new Thread(validateMonitor, new Runnable() {

                public void run() {
                    if (paintMonitor == null) {
                        return;
                    }
                    boolean interrupt_ = false;
                    try {
                        final CoalescedThreadsMonitor monitor = paintMonitor;
                        synchronized (monitor.getMonitor(true)) {
                            if (painting) {
                                System.err.println("Sprite " + base + " validate Thread's waitin' for paint...");
                            }
                            while (painting) {
                                System.err.print(".");
                                monitor.waitOnMonitor(10);
                            }
                            validating = true;
                            System.err.print("Sprite " + base + " is validating...");
                            if (spm instanceof SpritesCacheManager) {
                                spm.memorySensitiveCallback("loadResource", Sprite.this, new Object[]{}, new Class[]{});
                            } else {
                                loadResource();
                            }
                            System.err.println("done.");
                            statusImage();
                            final CoalescedThreadsMonitor monitor1 = imageSynch;
                            synchronized (monitor1.getMonitor(false)) {
                                System.err.println("OK");
                                if (spm instanceof SpritesCacheManager) {
                                    data = (Image) spm.memorySensitiveCallback("resizeData", Sprite.this, new Object[]{size, perspective, transforming, mirroring, scaling, obs}, new Class[]{Dimension.class, PerspectiveTransform.class, AffineTransform.class, AffineTransform.class, AffineTransform.class, Component.class});
                                } else {
                                    data = resizeData(size, perspective, transforming, mirroring, scaling, obs);
                                }
                                switch (cache) {
                                    case BUFFERED:
                                        if (data instanceof BufferedImage) {
                                            if (((BufferedImage) data).getType() == _type) {
                                                break;
                                            }
                                        }
                                        if (spm instanceof SpritesCacheManager) {
                                            data = (Image) spm.memorySensitiveCallback("convertToBuffered", Sprite.this, new Object[]{data, obs}, new Class[]{Image.class, Component.class});
                                        } else {
                                            data = convertToBuffered(data, obs);
                                        }
                                        break;
                                    case VOLATILE:
                                        if (data instanceof VolatileImage) {
                                            break;
                                        }
                                        if (spm instanceof SpritesCacheManager) {
                                            data = (Image) spm.memorySensitiveCallback("convertToVolatile", Sprite.this, new Object[]{data, obs}, new Class[]{Image.class, Component.class});
                                        } else {
                                            data = convertToVolatile(data, obs);
                                        }
                                        break;
                                    default:
                                        break;
                                }
                                if (compositeEnabled && !composited && data instanceof Image) {
                                    waitFor(trackerPty);
                                    if (data == null) {
                                        throw new NullPointerException("Sprite " + base + " had no data to draw ");
                                    }
                                    Graphics g1 = (cache == BUFFERED) ? data.getGraphics() : ((VolatileImage) data).createGraphics();
                                    g1.setClip(0, 0, data.getWidth(obs), data.getHeight(obs));
                                    GraphicsJAI g = createGraphicsJAI(g1, obs);
                                    Paint pnt = g.getPaint();
                                    Composite cps = g.getComposite();
                                    Color clr = g.getColor();
                                    if (Sprite.this.pnt != null) {
                                        g.setPaint(Sprite.this.pnt);
                                    }
                                    if (Sprite.this.clr != null) {
                                        g.setColor(Sprite.this.clr);
                                    }
                                    if (Sprite.this.cps != null) {
                                        g.setComposite(Sprite.this.cps);
                                    }
                                    g.fillRect(g.getClipBounds().x, g.getClipBounds().y, g.getClipBounds().width, g.getClipBounds().height);
                                    if (pnt != null) {
                                        g.setPaint(pnt);
                                    }
                                    if (cps != null) {
                                        g.setComposite(cps);
                                    }
                                    if (clr != null) {
                                        g.setColor(clr);
                                    }
                                    composited = true;
                                }
                                valid = true;
                                monitor1.notifyOnMonitor();
                            }
                            monitor.notifyOnMonitor();
                        }
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                        interrupt_ = true;
                    } catch (Throwable e) {
                        e.printStackTrace();
                    } finally {
                        final CoalescedThreadsMonitor monitor1 = validateMonitor;
                        synchronized (monitor1.getMonitor(false)) {
                            validating = false;
                            monitor1.notifyAllOnMonitor();
                        }
                        if (interrupt_) {
                            Thread.currentThread().interrupt();
                            return;
                        }
                    }
                }
            }, "T-Sprite validate");
            t.setPriority(Thread.MAX_PRIORITY);
            t.setDaemon(false);
            t.start();
        } else {
            valid = true;
        }
    }

    /***/
    public void addIIOReadProgressListener(IIOReadProgressListener l) {
        rProgList.add(l);
    }

    /***/
    public void addIIOReadWarningListener(IIOReadWarningListener l) {
        rWarnList.add(l);
    }

    /***/
    public void removeIIOReadProgressListener(IIOReadProgressListener l) {
        rProgList.remove(l);
    }

    /***/
    public void removeIIOReadWarningListener(IIOReadWarningListener l) {
        rWarnList.remove(l);
    }

    /***/
    public void addIIOWriteProgressListener(IIOWriteProgressListener l) {
        wProgList.remove(l);
    }

    /***/
    public void addIIOWriteWarningListener(IIOWriteWarningListener l) {
        wWarnList.remove(l);
    }

    /***/
    public void removeIIOWriteProgressListener(IIOWriteProgressListener l) {
        wProgList.remove(l);
    }

    /***/
    public void removeIIOWriteWarningListener(IIOWriteWarningListener l) {
        wWarnList.remove(l);
    }

    /***/
    public static GraphicsJAI createGraphicsJAI(Graphics g, Component obs) {
        GraphicsJAI jai = (g instanceof GraphicsJAI) ? (GraphicsJAI) g : GraphicsJAI.createGraphicsJAI((Graphics2D) g, obs);
        jai.setClip(g.getClip());
        jai.setRenderingHints(_getRenderingHints());
        return jai;
    }

    /***/
    public boolean isInnerResourceModeEnabled() {
        return isResource();
    }

    /***/
    public void setInnerResourceModeEnabled(boolean b) {
        setInnerResourceEnabled(b);
    }

    /** return last calaculted FPS
     * @return FPS string, a float-point number with 2 decimals
     * @see #markFPS()*/
    public String getFPS() {
        Formatter f = new Formatter();
        f.format("%1$.2f", lastFPS);
        return f.toString();
    }

    /** mark for FPS calculation
     * @return calculated FPS*/
    protected double markFPS() {
        double fps_n = 0;
        long newFrameTime = 0;
        fps_n = (double) (1000.0f / (float) ((newFrameTime = System.currentTimeMillis()) - lastFrameTime));
        fps_n = (double) (0.5f * (float) (lastFPS + fps_n));
        lastFPS = fps_n;
        lastFrameTime = newFrameTime;
        return fps_n;
    }

    public boolean isMultiThreadingEnabled() {
        return false;
    }

    public void setMultiThreadingEnabled(boolean b) {
    }

    public boolean isResourceLoaded() {
        return mt.checkID(trackerPty);
    }
    /** last mesured FPS (frames per second)*/
    protected double lastFPS = 0.0;
    /** last drawn frametime in millisec */
    protected long lastFrameTime = 0;

    /***/
    public void setThreadGroup(ThreadGroup tg) {
        valid = false;
        painting = false;
        validating = false;
        packing = false;
        if (imageSynch instanceof ThreadGroup) {
            imageSynch.interrupt();
        }
        if (paintMonitor instanceof ThreadGroup) {
            paintMonitor.interrupt();
        }
        if (packMonitor instanceof ThreadGroup) {
            packMonitor.interrupt();
        }
        if (validateMonitor instanceof ThreadGroup) {
            validateMonitor.interrupt();
        }
        if (tg instanceof ThreadGroup) {
            imageSynch = new CoalescedThreadsMonitor(tg, "TG-" + getClass().getName() + "-main " + base);
            validateMonitor = new CoalescedThreadsMonitor(tg, "TG-" + getClass().getName() + "-validate " + base);
            packMonitor = new CoalescedThreadsMonitor(tg, "TG-" + getClass().getName() + "-pack " + base);
            paintMonitor = new CoalescedThreadsMonitor(tg, "TG-" + getClass().getName() + "-paint " + base);
        } else {
            imageSynch = new CoalescedThreadsMonitor("TG-" + getClass().getName() + "-main " + base);
            validateMonitor = new CoalescedThreadsMonitor("TG-" + getClass().getName() + "-validate " + base);
            packMonitor = new CoalescedThreadsMonitor("TG-" + getClass().getName() + "-pack " + base);
            paintMonitor = new CoalescedThreadsMonitor("TG-" + getClass().getName() + "-paint " + base);
        }
        imageSynch.setCoalesceEnabled(false);
        validateMonitor.setCoalesceEnabled(false);
        packMonitor.setCoalesceEnabled(false);
        paintMonitor.setCoalesceEnabled(true);
        validateMonitor.setMaxPriority(Thread.MAX_PRIORITY);
        paintMonitor.setMaxPriority(validateMonitor.getMaxPriority() - 1);
        packMonitor.setMaxPriority(paintMonitor.getMaxPriority() - 1);
    }

    /***/
    public ThreadGroup getThreadGroup() {
        return validateMonitor.getParent();
    }
}
