package net.library.jiga.sf3;

import java.applet.Applet;
import java.applet.AudioClip;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.VolatileImage;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.lang.ref.ReferenceQueue;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import javax.media.jai.GraphicsJAI;
import javax.media.jai.PerspectiveTransform;
import javax.swing.Timer;
import javazoom.jl.decoder.JavaLayerException;
import javazoom.jl.player.Player;
import net.library.jiga.sf3.system.CoalescedThreadsMonitor;
import net.library.jiga.sf3.system.SpritesCacheManager;
import net.library.jiga.sf3.system.Threaded;

/**
 * An extended Sprite to handle animations in Java Environnement.
 * The implemented timer is the most important part of an animation design pattern. It must implement a real time iterator to avoid visual artefacts while drawing
 * the animation on screen. Yet correctly Threaded, the timer, a.k.a animator, calls a small timing framework and loads up to 1 second of consecutive frames
 * in the soft-cache. So there's also a buffer, that can be ranged to a specific buffering-time in milliseconds (ms).
 * @see #loadBuffer(long, boolean)
 * @see Sprite
 */
public class Animation extends Sprite implements Iterator, Runnable, Serializable, Threaded {

    /** serial version UID to identify serialize operations */
    private static final long serialVersionUID = 2323;
    /** starting frame n */
    protected int startingFrame;
    /** prefix to each frame file */
    protected String prefix;
    /** suffix to each frame file */
    protected String suffix;
    /** current frame index */
    protected int animator;
    /** reverse playing
     * @default false*/
    protected boolean reverse = false;
    /** timer */
    protected transient Timer timer;
    /** frames sorted map */
    protected transient SortedMap<Integer, Sprite> frames;
    /** cache map */
    protected final SpritesCacheManager<Integer, Sprite> spm;
    /***/
    protected final SortedMap<Integer, File> imageFiles = Collections.synchronizedSortedMap(new TreeMap<Integer, File>());
    /** sound fx */
    protected transient Sound sfx;
    /** sound fx file path */
    protected String sfx_path;
    /** sound directory path */
    protected String sfx_dir = ".cache" + File.separator + "soundIO";
    /** start time stamp (ms) */
    protected long start = 0;
    /** last timer tick time stamp (ms)*/
    protected long lastTick = 0;
    /** current state
     * @see #PLAYING
     * @see #STOPPED
     * @see #PAUSED */
    protected int statusID;
    /** compressed cache (a bit slow on low-end systems)
     * @default false*/
    protected boolean compress = false;
    /** file swapped cache map (enabled is recommended for large amount of frames)
     * @default false*/
    protected boolean swap = false;
    /** BufferedImage dis/enabled
     * @see BufferedImage
     * @see VolatileImage
     * @see Sprite#cache
     * @default true*/
    protected boolean buffered = true;
    /** transforms dis/enabled
     * @default true*/
    protected boolean transform = true;
    /** playing state */
    public final static int PLAYING = 0;
    /** paused state */
    public final static int PAUSED = 1;
    /** stopped state*/
    public final static int STOPPED = 2;
    /** frames amount count */
    public int length;
    /** current framerate mesured in ms, default is 24FPS
     * @see #timer*/
    public int frameRate = (int) (1000.0f / 24.0f); // milliseconds
    /** last drawn frame index */
    protected int lastFramePosition = 0;
    /** next constant value, used for iterating
     * @see #getAnimatorValue(int, int)*/
    protected final int NEXT = 0;
    /** previous constant value, used for iterating
     * @see #getAnimatorValue(int, int)*/
    protected final int PREVIOUS = 1;
    /** automated cache cleanup (not recommended)
     * @default false*/
    public boolean auto = false;
    /** sound inner-resource dis/enabled */
    private boolean sfxIsRsrc;

    /**
     * Initializes an animation with the specified MediaTracker to handle loading.
     * @see sf3.system#Resource
     * @see Sprite#mime
     * @see Sprite#size
     * @discussion (comprehensive description)
     * @param startingFrame starting frame number that appears in frame sprites file names
     * @param endingFrame ending frame number that appears in frame sprites file names
     * @param baseLink string path to base directory of the frame sprites
     * @param prefix string prefix to every frame sprites filenames
     * @param suffix string suffix to every frame sprites filenames (usually file .extension)
     * @param format image mime type of Sprites
     * @param size desired image size of Sprites
     * @param rsrcMode inner-resource mode for frame sprites files dis/enabled
     * @throws java.net.URISyntaxException if the base path does not exist
     */
    public Animation(String baseLink, boolean rsrcMode, int startingFrame, int endingFrame, String prefix, String suffix, String format, Dimension size) throws URISyntaxException {
        super(false, baseLink, rsrcMode, format, size, true);
        sfxIsRsrc = rsrcMode;
        this.length = endingFrame - startingFrame + 1;
        frames = Collections.synchronizedSortedMap(spm = new SpritesCacheManager<Integer, Sprite>(length));
        spm.setCompressionEnabled(compress);
        spm.setSwapDiskEnabled(swap);
        spm.setAutoCleanupEnabled(auto);
        this.startingFrame = startingFrame;
        this.prefix = prefix;
        this.suffix = suffix;
        animator = (reverse) ? length - 1 : 0;
    }

    /**
     * Initializes the animation with blank frames that can be drawn on using the getImage() method on each Sprite.
     * @see #getImage(Component)
     *
     * @param length amount of frames to put in this Animation
     * @param size desired dimension of each sprite image
     * @param format sprite image mime type to use (e.g. image/jpeg)
     * @see #loadBlank(int)
     * @discussion (comprehensive description)
     */
    public Animation(int length, String format, Dimension size) {
        super(false, createBufferedImage(size, Sprite.DEFAULT_TYPE), format, size);
        sfxIsRsrc = innerResource;
        this.length = length;
        this.frames = Collections.synchronizedSortedMap((spm = new SpritesCacheManager<Integer, Sprite>(length)));
        spm.setCompressionEnabled(compress);
        spm.setSwapDiskEnabled(swap);
        spm.setAutoCleanupEnabled(auto);
        animator = (reverse) ? length - 1 : 0;
    }

    /**
     * method that clears the resources on finalization
     * @see #clearResource()
     * @throws java.lang.Throwable thrown by the super class
     */
    public void finalize() throws Throwable {
        stop();
        frames.clear();
        super.finalize();
    }

    /** Serialize write method, current state and resources are stored, but the MediaTracker and the Component
     * @param out some ObjectOutputStream to write into. */
    private void writeObject(ObjectOutputStream out) throws IOException {
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        stop();
        spm.setSwapDiskEnabled(swap = true);
        System.out.println("*            Animation's serializing...");
        loadResource();
        out.defaultWriteObject();
        synchronized (imageFiles) {
            for (Iterator<Integer> i = imageFiles.keySet().iterator(); i.hasNext();) {
                int key;
                File f = imageFiles.get(key = i.next());
                out.writeInt(key);
                out.writeObject(f);
                if (f.exists()) {
                    RandomAccessFile raf = new RandomAccessFile(f, "r");
                    out.writeLong(raf.length());
                    byte[] b = new byte[512];
                    int readBytes = 0;
                    while ((readBytes = raf.read(b)) != -1) {
                        out.write(b, 0, readBytes);
                    }
                    System.out.println("Animation " + base + " has written an image of " + raf.length() + " bytes long.");
                    raf.close();
                } else {
                    out.writeLong(0L);
                }
            }
        }
        sfx = initSound();
        if (sfx.load(sfx_path)) {
            out.writeBoolean(true);
            BufferedInputStream bis = (innerResource) ? new BufferedInputStream(getClass().getResourceAsStream(sfx_path)) : new BufferedInputStream(new FileInputStream(sfx_path));
            File d = new File(sfx_dir);
            d.mkdirs();
            File sfxFile = File.createTempFile("sfx_", "" + hash, d);
            sfxFile.deleteOnExit();
            out.writeObject(sfxFile);
            RandomAccessFile raf = new RandomAccessFile(sfxFile, "rws");
            byte[] b = new byte[512];
            int readBytes = 0;
            while ((readBytes = bis.read(b)) != -1) {
                raf.write(b, 0, readBytes);
            }
            out.writeLong(sfxFile.length());
            raf.seek(0);
            b = new byte[512];
            while (raf.read(b) != -1) {
                out.write(b);
            }
            System.out.println("Animation " + base + " has written a sound of " + raf.length() + " bytes long.");
            raf.close();
            bis.close();
        } else {
            out.writeBoolean(false);
        }
        System.out.println("*            Animation's been serialized.");
        Thread.currentThread().setPriority(pty);
    }

    /** Serialize read method, current state and resources are recovered
     * @param in some ObjectInputStream to read from.*/
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        System.out.println("*            Animation " + base + " is deserializing...");
        in.defaultReadObject();
        frames = Collections.synchronizedSortedMap(spm);
        synchronized (imageFiles) {
            for (Iterator<Integer> i = imageFiles.keySet().iterator(); i.hasNext();) {
                int key, rkey;
                File f, rf;
                f = imageFiles.get(key = i.next());
                rkey = in.readInt();
                rf = (File) in.readObject();
                if (key != rkey) {
                    throw new IOException("wrong key in Animation " + base + " image files base : " + key + " read : " + rkey);
                }
                if (!f.equals(rf)) {
                    throw new IOException("wrong file in Animation " + base + " image files base : " + f + " read : " + rf);
                }
                long len;
                if ((len = in.readLong()) > 0) {
                    if (!f.exists()) {
                        RandomAccessFile raf = new RandomAccessFile(f, "rws");
                        byte[] b = new byte[(int) 512];
                        int readBytes = 0;
                        int rBytes;
                        boolean fread = true;
                        do {
                            if (readBytes + b.length > len) {
                                in.readFully(b, 0, rBytes = (int) (len - readBytes));
                            } else {
                                rBytes = in.read(b);
                            }
                            System.out.print(".");
                            if (rBytes != -1) {
                                raf.write(b, 0, rBytes);
                                readBytes += rBytes;
                            } else {
                                fread = false;
                            }
                            if (readBytes >= len) {
                                fread = false;
                            }
                        } while (fread);
                        System.out.println("Animation's read " + readBytes + " bytes out of " + len);
                        raf.close();
                    } else {
                        in.skipBytes((int) len);
                    }
                    f.deleteOnExit();
                }
            }
        }
        if (in.readBoolean()) {
            File dir = new File(sfx_dir);
            dir.mkdirs();
            File _sfx = (File) in.readObject();
            if (_sfx == null) {
                _sfx = File.createTempFile("sfx_", "" + hash, dir);
            }
            _sfx.deleteOnExit();
            long len = in.readLong();
            if (!_sfx.exists() || _sfx.length() != length) {
                FileOutputStream fos = new FileOutputStream(new RandomAccessFile(_sfx, "rws").getFD());
                byte[] b = new byte[(int) 512];
                int readBytes = 0;
                int rBytes;
                boolean fread = true;
                do {
                    if (readBytes + b.length > len) {
                        in.readFully(b, 0, rBytes = (int) (len - readBytes));
                    } else {
                        rBytes = in.read(b);
                    }
                    System.out.print(".");
                    if (rBytes != -1) {
                        fos.write(b, 0, rBytes);
                        readBytes += rBytes;
                    } else {
                        fread = false;
                    }
                    if (readBytes >= len) {
                        fread = false;
                    }
                } while (fread);
                fos.close();
                System.out.println("Animation serialized sfx path: " + sfx_path);
            } else {
                in.skip(len);
            }
            sfx_path = _sfx.getCanonicalPath();
            setSfx(sfx_path, false, sfx_frame);
        }
        System.out.println("*            Animation's been deserialized.");
        Thread.currentThread().setPriority(pty);
    }

    /** sets the animation duration which modifies the current framerate, e.g. 3000 ms for an amount 3 frames
     * will set the framerate to 1FPS for this animation.
     * @param millis the desired duration in ms */
    public void setRealTimeLength(long millis) {
        frameRate = (int) ((float) millis / (float) length);
    }

    /** returns duration based on framerate and frame length of the animation.
     * @return current duration in millis*/
    protected long realTimeLength() {
        return frameRate * length;
    }

    /** elapsed period of frames since the animator timer started
     * @return elapsed amount of frames (always positive or zero)
     * @see #elapsedTime()*/
    private int elapsedFrames() {
        return (int) Math.floor((float) elapsedTime() / (float) realTimeLength() * (float) length);
    }

    /** elapsed period of time since the last tick occured on animator timer
     * @return elapsed time in ms since last tick of the timer
     * @see #lastTick*/
    private long elapsedTickTime() {
        if (status() == PLAYING) {
            return System.currentTimeMillis() - lastTick;
        } else {
            return 0;
        }
    }

    /** elapsed frames since last tick occured on animator timer
     * @return elapsed amount of frames since last tick of the timer
     * @see #elapsedTickTime()*/
    private int elapsedTickFrames() {
        return (int) Math.floor((float) elapsedTickTime() / (float) realTimeLength() * (float) length);
    }

    /** re-adjusts startTime to the animation current timeframe, thereby the animator can loop. But use
     * play() to make loops is the correct manner.
     * @return new start time in ms if current is to old
     * @see #start
     * @see #play()*/
    protected long adjustStartTime() {
        if (elapsedTime() >= realTimeLength()) {
            return start = System.currentTimeMillis();
        } else {
            return start;
        }
    }

    /** loads and returns one blank frame with the specified index with the current settings (size, mime type, etc.)
     *@param i frame index (it is painted on the background of the sprite)
     * @return frame Sprite */
    private Sprite loadBlank(int i) {
        System.out.println("sf3.Animation : loading blank sprite at frame " + i);
        Image img = (!buffered) ? (Image) createVolatileImage(size) : createBufferedImage(size, _type);
        Graphics2D g = (Graphics2D) img.getGraphics();
        g.setFont(new Font("Arial", Font.ITALIC, 20));
        g.setColor(Color.BLACK);
        g.fillRect(0, 0, size.width, size.height);
        g.setColor(Color.GRAY);
        g.drawString("(" + (i + 1) + ")", (int) ((float) size.width / 2.0f), (int) ((float) size.height / 2.0f));
        Sprite sp = (!buffered) ? new Sprite((VolatileImage) img, mime, size) : new Sprite((BufferedImage) img, mime, size);
        return sp;
    }

    /** marks the FPS monitor to mesure the actual framerate
     * @return actual fps */
    protected double markFPS() {
        int newFramePosition = 0;
        double fps = 0;
        if ((newFramePosition = getPosition()) != lastFramePosition) {
            fps = super.markFPS();
            lastFramePosition = newFramePosition;
        }
        return fps;
    }

    /** instances a new Sound interface
     * @return one sound interface
     * @see Sound*/
    private Sound initSound() {
        return new Sound() {

            AudioClip bgSfx = null;
            Player player = null;
            boolean bgSfx_playing = false;
            boolean mp3Enabled = false;
            String fileRsrc = null;

            public boolean load(String filename) {
                if (filename == null || filename == "") {
                    return false;
                }
                if (fileRsrc != null) {
                    if (fileRsrc.equals(filename)) {
                        return true;
                    }
                }
                URL rsrc = null;
                try {
                    rsrc = (sfxIsRsrc) ? getClass().getResource(filename) : new File(filename).toURL();
                    bgSfx = Applet.newAudioClip(rsrc);
                    player = new Player((sfxIsRsrc) ? getClass().getResourceAsStream(filename) : new BufferedInputStream(new FileInputStream(filename)));
                } catch (JavaLayerException ex) {
                    ex.printStackTrace();
                } catch (MalformedURLException ex) {
                    ex.printStackTrace();
                } finally {
                    if (rsrc == null) {
                        return false;
                    }
                    fileRsrc = filename;
                    return true;
                }
            }

            public void loop() {
                if (mp3Enabled) {
                    bgSfx_playing = true;
                    Thread t_mp3 = new Thread(new Runnable() {

                        public void run() {
                            while (bgSfx_playing) {
                                try {
                                    if (!load(fileRsrc)) {
                                        stop();
                                        continue;
                                    }
                                    if (player instanceof Player) {
                                        player.play();
                                    }
                                } catch (JavaLayerException ex) {
                                    ex.printStackTrace();
                                }
                            }
                        }
                    }, "T-loop-mp3");
                    t_mp3.start();
                } else {
                    if (bgSfx instanceof AudioClip) {
                        bgSfx_playing = true;
                        bgSfx.loop();
                    }
                }
            }

            public void play() {
                assert fileRsrc != null : "Sound : you must call load(String) once before to play the Sound";
                if (mp3Enabled) {
                    if (!load(fileRsrc)) {
                        return;
                    }
                    Thread t_mp3 = new Thread(new Runnable() {

                        public void run() {
                            try {
                                bgSfx_playing = true;
                                if (player instanceof Player) {
                                    player.play();
                                }
                                bgSfx_playing = false;
                            } catch (JavaLayerException ex) {
                                ex.printStackTrace();
                            }
                        }
                    }, "T-play-mp3");
                    t_mp3.start();
                } else {
                    if (bgSfx instanceof AudioClip) {
                        bgSfx.play();
                        bgSfx_playing = true;
                    }
                }
            }

            public void stop() {
                if (mp3Enabled) {
                    if (player instanceof Player) {
                        player.close();
                    }
                    bgSfx_playing = false;
                } else {
                    if (bgSfx instanceof AudioClip) {
                        bgSfx.stop();
                    }
                    bgSfx_playing = false;
                }
            }
        };
    }

    /** calculates the hashcode of the animation
     * @return the hashcode */
    public int hashCode() {
        return super.hashCode() + ((frames != null) ? frames.hashCode() : 0);
    }

    /**
     * checks for equality with the specified object
     * @return true or false
     * @param o the specified object to check for equality
     */
    public boolean equals(Object o) {
        if (o instanceof Animation) {
            Animation a = (Animation) o;
            Sprite first0, first1, last0, last1;
            if (!frames.isEmpty() && !a.frames.isEmpty()) {
                first0 = frames.get(frames.firstKey());
                first1 = a.frames.get(a.frames.firstKey());
                last0 = frames.get(frames.lastKey());
                last1 = a.frames.get(a.frames.lastKey());
            } else {
                first0 = first1 = last0 = last1 = null;
            }
            return (((base != null && a.base != null) ? a.base.equals(base) : (base == null && a.base == null)) && ((size != null && a.size != null) ? a.size.equals(size) : (size == null && a.size == null)) && (first0 != null && first1 != null && last0 != null && last1 != null) ? (first0.equals(first1) && last0.equals(last1)) : (a.frames.isEmpty() && frames.isEmpty()));
        } else {
            return false;
        }
    }

    /** gets the status of the animation
     * @return status id
     * @see #STOPPED
     * @see #PLAYING
     * @see #PAUSED
     * @see #state*/
    public int status() {
        if (timer != null) {
            return statusID = (timer.isRunning()) ? PLAYING : PAUSED;
        } else {
            statusID = STOPPED;
        }
        return statusID;
    }

    /**
     *
     * @see #isBufferedImageEnabled()
     * @return true or false
     */
    public boolean isOpaque() {
        return !isBufferedImageEnabled();
    }

    /**
     *
     * @see #setBufferedImageEnabled(boolean)
     * @param b to dis/enable the opacity
     */
    public void setOpaque(boolean b) {
        setBufferedImageEnabled(!b);
    }

    /** returns true if the BufferedImage is used for the sprites.
     * @return true if the BufferedImage is used for the sprites
     * @see #setBufferedImageEnabled(boolean)*/
    public boolean isBufferedImageEnabled() {
        return buffered;
    }

    /** activates BufferedImage for the sprites
     * @param b enabled
     * @see Sprite#cache*/
    public void setBufferedImageEnabled(boolean b) {
        buffered = b;
    }

    /** activates compressed cache map
     * @param b compressed cachee enabled
     * @see SpritesCacheManager#setCompressionEnabled(boolean)*/
    public void setCompressedCacheEnabled(boolean b) {
        compress = b;
        spm.setCompressionEnabled(b);
    }

    /** activates swap disk cache support, which is always enabled if the animation has been serialized by an OutputStream.
     * @param b swapped cache enabled
     * @see SpritesCacheManager#setSwapDiskEnabled(boolean)*/
    public void setSwapDiskCacheEnabled(boolean b) {
        swap = b;
        spm.setSwapDiskEnabled(b);
    }

    /** gets the state of the current animation
     * @return current state
     * @see #status()*/
    public int getState() {
        return status();
    }

    /** gets the current allocation size of cache
     * @return current allocation size in percent
     * @see SpritesCacheManager#allocSize()*/
    public double allocSize() {
        return spm.allocSize();
    }

    /** gets the animation length
     * @return frames amount of this animation
     * @see #length*/
    public int length() {
        return length;
    }

    /** positions the animator to corresponding frame index
     * @param i frame index position
     * @return timeframe position in ms
     * @see #getPosition()*/
    public long position(int i) {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            assert (i < length && i >= 0) : getClass().getCanonicalName() + " iterator is not in the correct interval!";
            animator = i;
            monitor.notifyOnMonitor();
        }
        adjustStartTime();
        status();
        return getTimeFramePosition();
    }

    /** gets the current position of the animator
     * @return current animator value corresponding to frame index*/
    public int getPosition() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            monitor.notifyOnMonitor();
            return animator;
        }
    }

    /**
     * refreshes the animation and the cache map to current params and prints current iterator value.
     * @discussion (comprehensive description)
     * @see #spm
     * @see #setZoomEnabled(boolean, double)
     * @see #setFlipEnabled(boolean, int)
     * @see Sprite#validate()
     */
    public void validate() {
        super.validate();
        spm.ensureListCapacity(spm.getInitialListCapacity());
        spm.setCompressionEnabled(compress);
        spm.setSwapDiskEnabled(swap);
        setZoomEnabled(transform, zoom);
        setFlipEnabled(transform, mirror);
    }

    /** returns the current frame sprite that the animator is pointing on
     * @see Sprite
     * @return current frame sprite*/
    public Sprite getCurrentSprite() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            monitor.notifyOnMonitor();
            System.out.println("current sprite called, f n¡ is " + animator);
            return getSprite(animator);
        }
    }

    /***/
    public Sprite getSprite(int index) {
        Sprite sp = null;
        if ((sp = frames.get(index)) == null) {
            sp = loadError(index);
        }
        return refreshSpriteData(sp, (reverse) ? length - index : index);
    }

    /** refreshes the specified Sprite instance to the current settings
     * @param sp Sprite instance to refresh
     * @param id the id on which the MediaTracker will track
     * @return the refreshed Sprite instance*/
    private Sprite refreshSpriteData(Sprite sp, int id) {
        System.out.println("Animation's refreshing " + sp.base + "...");
        sp.rProgList = rProgList;
        sp.rWarnList = rWarnList;
        sp.wProgList = wProgList;
        sp.wWarnList = wWarnList;
        validate();
        sp._type = _type;
        sp.setStoreImageEnabled(!spm.isSwapDiskEnabled());
        sp.setMt(mt, obs);
        sp.setPreferredSize(size);
        sp.setSize(size);
        if (transform != sp.isZoomEnabled() || sp.getZoomValue() != zoom) {
            sp.setZoomEnabled(transform, zoom);
        }
        if (transform != sp.isFlipEnabled() || sp.getMirrorValue() != mirror) {
            sp.setFlipEnabled(transform, mirror);
        }
        if(transform) {
            sp.setTX(transforming);
            sp.setPTX(perspective);
        }
        sp.setCompositeEnabled(compositeEnabled);
        sp.setComposite(cps);
        sp.setPaint(pnt);
        sp.setColor(clr);
        sp.setTrackerPty(id);
        sp.setSPM(spm);
        sp.validate();
        return sp;
    }

    /**
     * loads the frame sprites indexed by the specified number
     * @discussion (comprehensivedescription)
     * @param i index of the frame
     * @return frame sprite loaded at index i
     */
    private Sprite loadFrame(int i) {
        Sprite sp = null;
        spm.ensureListCapacity(spm.getInitialListCapacity());
        if (!frames.containsKey(i)) {
            System.err.print("OK\n\r");
            if (base != null) {
                String path = src + ((innerResource) ? "/" : File.separator) + prefix + (int) (startingFrame + i) + suffix;
                System.out.println("loading frame: " + ((innerResource) ? getClass().getResource(path) : path));
                try {
                    sp = new Sprite(path, innerResource, mime, size, buffered);
                } catch (URISyntaxException ex) {
                    ex.printStackTrace();
                    sp = loadError(i);
                }
            } else {
                sp = loadBlank(i);
            }
            sp = refreshSpriteData(sp, (reverse) ? length - i : i);
            frames.put(i, sp);
            if (sp.src instanceof File) {
                imageFiles.put(i, (File) sp.src);
            }
            spm.ensureListCapacity(spm.getInitialListCapacity());
            /*if(swap)
            sp.clearResource();*/
            return sp;
        }
        sp = frames.get(i);
        sp = refreshSpriteData(sp, (reverse) ? length - i : i);
        spm.ensureListCapacity(spm.getInitialListCapacity());
        /* if(swap)
        sp.clearResource();*/
        return sp;
    }

    /** loads and returns a frame sprite instance for error at index i
     * @param i the index to give error at
     * @return frame sprite instance */
    private Sprite loadError(int i) {
        Image img = (!buffered) ? (Image) createVolatileImage(size) : createBufferedImage(size, _type);
        GraphicsJAI g = createGraphicsJAI(img.getGraphics(), this);
        g.setFont(new Font("Arial", Font.ITALIC, 20));
        g.setColor(Color.RED);
        g.fillRect(0, 0, size.width, size.height);
        g.setColor(Color.GRAY);
        g.drawString("error_" + i, (int) ((float) size.width / 2.0f), (int) ((float) size.height / 2.0f));
        Sprite sp = (!buffered) ? new Sprite((VolatileImage) img, mime, size) : new Sprite((BufferedImage) img, mime, size);
        return sp;
    }

    /**
     * unloads the frame sprite instance indexed with the argument.
     * this sprite instance can be recovered if the isSwapDiskCacheEnabled() returns true.
     * @discussion (comprehensive description)
     * @param i index of the frame sprite instance
     */
    protected void unloadFrame(int i) {
        if (frames.containsKey(i)) {
            try {
                spm.memorySensitiveCallback("memoryClear", spm, new Object[]{frames.get(i)}, new Class[]{Object.class});
            } catch (Throwable ex) {
                ex.printStackTrace();
            }
        }
    }

    /** elapsed period of time since the animator timer started
     * @return elapsed time since started in ms
     * @see #realTimeLength()*/
    private long elapsedTime() {
        long t = System.currentTimeMillis() - start;
        if (t > realTimeLength()) {
            if (timer != null) {
                timer.stop();
            }
        //t = lastTick-start;
        }
        return t;
    }

    /** current time-frame position
     * @return current time-frame position in ms
     * @see #getPosition()
     * @see #realTimeLength()*/
    public long getTimeFramePosition() {
        return (long) (((float) getPosition() / (float) length) * (float) realTimeLength());
    }

    /**
     * positions the animator to the corresponding time-frame
     * @return corresponding new positioned frame index value
     * @see #position(int)
     * @param timeFrame time frame position in ms (must be positive and less than the total real length time)
     * @see #realTimeLength()
     */
    public int position(long timeFrame) {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (!reverse) {
                animator = Math.min(length - 1, Math.max(0, (int) (((float) timeFrame / (float) realTimeLength()) * (float) length)));
            } else {
                animator = Math.min(length - 1, Math.max(0, length - 1 - (int) (((float) timeFrame / (float) realTimeLength()) * (float) length)));
            }
            position(animator);
            monitor.notifyOnMonitor();
            return animator;
        }
    }

    /**
     * draws the animation on the specified graphics and component context
     * @return true or false whether the drawing's completed or not (this value cannot be used for synchronization)
     * @see #draw(Component, Graphics2D, AffineTransform)
     * @param g2 graphics instance to draw on
     * @param obs the component that is observing this animation
     * @throws java.lang.InterruptedException if the current Thread gets interrupted
     */
    public boolean draw(Component obs, Graphics2D g2) throws InterruptedException {
        return draw(obs, g2, null, null);
    }
    /** the current sprite instance that this animation is using */
    private transient Sprite currentSprite = null;

    /**
     * draws with the specified AffineTransform instance
     * @return true or false whether the drawing's completed or not (this value cannot be used for synchronization)
     * @param g2 graphics instance to draw on
     * @param tx transform instance
     * @param obs the component that is observing this animation
     * @throws java.lang.InterruptedException if the current Thread gets interrupted
     */
    public boolean draw(Component obs, Graphics2D g2, AffineTransform tx, PerspectiveTransform ptx) throws InterruptedException {
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(paintMonitor.getMaxPriority());
        boolean interrupt_ = false;
        boolean complete = false;
        spm.ensureListCapacity(spm.getInitialListCapacity());
        try {
            final CoalescedThreadsMonitor monitor0 = validateMonitor;
            synchronized (monitor0.getMonitor(false)) {
                if (validating) {
                    System.err.println(base + " Animation's waitin' for buffering...");
                }
                while (validating) {
                    System.err.print(".");
                    monitor0.waitOnMonitor(10);
                }
                painting = true;
                System.err.print("OK\n\r");
                complete = (Boolean) spm.memorySensitiveCallback("draw", currentSprite = getCurrentSprite(), new Object[]{obs, g2, tx, ptx}, new Class[]{Component.class, Graphics2D.class, AffineTransform.class, PerspectiveTransform.class});
                markFPS();
                monitor0.notifyOnMonitor();
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
            interrupt_ = true;
        } catch (Throwable t) {
            t.printStackTrace();
            interrupt_ = true;
        } finally {
            final CoalescedThreadsMonitor monitor1 = paintMonitor;
            synchronized (monitor1.getMonitor(false)) {
                painting = false;
                monitor1.notifyAllOnMonitor();
            }
            Thread.currentThread().setPriority(pty);
            spm.ensureListCapacity(spm.getInitialListCapacity());
            if (interrupt_) {
                throw new InterruptedException("Animation " + base + " caught an interruption.");
            }
            return complete;
        }
    }

    /**
     * sets the animation SFX to play.
     * @see #sfx
     * @see #initSound()
     *@param frameMark frame index telling when to play the sfx
     * @param sfx_path full resource path to the sound file (currently supporting mpeg-layer-3 or PCM wave files)
     * @param rsrcMode specifies the inner-resource mode dis/enabled
     * @see sf3.system#Resource
     */
    public void setSfx(String sfx_path, boolean rsrcMode, int frameMark) {
        sfxIsRsrc = rsrcMode;
        assert (frameMark < length - 1 && frameMark >= 0) : getClass().getCanonicalName() + " frame mark for sfx is not in the correct interval!";
        sfx_frame = frameMark;
        sfx = initSound();
        if (sfx_path != null) {
            sfx.load(sfx_path);
        }
        this.sfx_path = sfx_path;
    }

    /**
     * sets whether to reverse or not this animation when playing it
     * @param b true to reverse animation
     */
    public void setReverseEnabled(boolean b) {
        reverse = b;
    }
    /** tells which frame index to use to play the associated sound FX
     * @see #sfx
     * @see #initSound() */
    int sfx_frame = 0;

    /**
     * the animator run method. please use the play() method instead !
     * @see #play()*/
    public void run() {
        if (paintMonitor == null) {
            return;
        }
        boolean interrupt_ = false;
        try {
            final CoalescedThreadsMonitor monitor0 = paintMonitor;
            synchronized (monitor0.getMonitor(true)) {
                boolean sound = (sfx_path != null) ? sfx.load(sfx_path) : false;
                if (painting) {
                    System.err.println("Animator " + base + " Thread's waitin' for buffer...");
                }
                while (painting) {
                    System.err.print(".");
                    monitor0.waitOnMonitor(10);
                }
                validating = true;
                final CoalescedThreadsMonitor monitor2 = imageSynch;
                synchronized (monitor2.getMonitor(false)) {
                    System.err.print("OK\n\r");
                    System.err.print("f n¡ : " + animator + " -> ");
                    if (reverse) {
                        animator = length - elapsedFrames();
                    } else {
                        animator = elapsedFrames();
                    }
                    animator = Math.min(animator, length - 1);
                    animator = Math.max(animator, 0);
                    System.err.print(animator + "\n\r");
                    lastTick = System.currentTimeMillis();
                    if (sfx_frame == animator && sfx instanceof Sound && sound) {
                        sfx.play();
                    }
                    monitor2.notifyOnMonitor();
                }
                monitor0.notifyOnMonitor();
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
            interrupt_ = true;
        } finally {
            final CoalescedThreadsMonitor monitor1 = validateMonitor;
            synchronized (monitor1.getMonitor(false)) {
                validating = false;
                monitor1.notifyAllOnMonitor();
            }
            if (interrupt_) {
                Thread.currentThread().interrupt();
            }
        }
    }

    /**
     * returns the current frame image instance, seen by the current Component observer.
     * @return the current frame image instance
     * @see #obs
     * @param obs the component that will observe the image
     * @throws java.lang.InterruptedException if the current Thread gets interrupted
     */
    public Image getImage(Component obs) throws InterruptedException {
        return getCurrentSprite().getImage(obs);
    }

    /**
     * returns the synchronized frames map of this animation
     * @return a synchronized frames map view
     */
    public Map<Integer, Sprite> getFrames() {
        return frames;
    }

    /** tells whether mirror transform is enabled
     * @see #getMirrorOrientation()
     * @return true or false*/
    public boolean isMirrored() {
        return (mirror == NONE) ? false : true;
    }

    /** returns the current mirror orientation or
     * @see #NONE
     * @see #HORIZONTAL
     * @see #VERTICAL
     * @see #mirror
     * @return mirror orientation */
    public int getMirrorOrientation() {
        return mirror;
    }

    /** tells whether the zoom transform is enabled
     * @see #getZoomValue()
     * @return true or false*/
    public boolean isZoomed() {
        return (zoom == 1.0) ? false : true;
    }

    /** returns current zoom value
     * @return zoom value (1.0 is no zoom)*/
    public double getZoomValue() {
        return zoom;
    }

    /** sets the current framerate to the specified delay
     * @param delay the delay to use as framerate in ms
     * @throws NullPointerException if the specified delay is zero*/
    public void setFrameRate(int delay) {
        try {
            if (delay < 0) {
                throw new NullPointerException("framerate delay must be different from Zero!");
            }
            frameRate = delay;
        } catch (NullPointerException e) {
            e.printStackTrace();
        }
    }

    /** returns the timestamp used by the animator timer when it started *
     * @return the timestamp got when started playing
     * @see #start
     * @see #play()*/
    public long getStartTime() {
        return start;
    }
    /** the current Thread running the animator
     * @see #run()*/
    private transient Thread play = null;

    /**
     * starts playing this animation. Actually the animator is incremented or decreased, depending on reverse status.
     * if it is called more than once it will safely manage to get it to continue (start time will be updated !) even if it has been stopped or paused.
     * The sound FX will be played when the animator comes to set frame index for the sound if any available.
     * @see #sfx_frame
     * @see #setSfx(String, boolean, int)
     */
    public void play() {
        System.out.println(start);
        start = adjustStartTime();
        switch (status()) {
            case PAUSED:
                System.out.println("resumed>");
            case STOPPED:
                rewind();
                timer = new Timer(23, new ActionListener() {

                    public void actionPerformed(ActionEvent e) {
                        if (play instanceof Thread) {
                            if (play.isAlive()) {
                                return;
                            }
                        }
                        Thread t = play = new Thread(validateMonitor, Animation.this, "T-Animation-Play");
                        t.setDaemon(false);
                        t.setPriority(Thread.MAX_PRIORITY);
                        t.start();
                    }
                });
                timer.start();
                break;
            case PLAYING:
                break;
            default:
                break;
        }
        System.out.println("play>");
    }

    /**
     * pauses this animation. animator timer is stopped. next time play() is called it will restart at the current animator index.
     */
    public void pause() {
        if (timer != null) {
            timer.stop();
        }
        System.out.println("pause||");
    }

    /**
     * stops playing the animation. the animator will be reset. the animator timer is cancelled.
     */
    public void stop() {
        pause();
        timer = null;
        System.out.println("stop[]");
    }

    /** returns the animator for the specified operation-mode
     * @param mode operation-mode
     * @see #NEXT
     * @see #PREVIOUS
     * @param position where to get the iterator from
     * @return the animator requested value (current animator won't be modified)*/
    private int getAnimatorValue(int mode, int position) {
        int i = position;
        switch (mode) {
            case NEXT:
                if (reverse) {
                    if (i > 0) {
                        i--;
                    } else {
                        i = length - 1;
                    }
                } else {
                    if (i < length - 1) {
                        i++;
                    } else {
                        i = 0;
                    }
                }
                break;
            case PREVIOUS:
                if (reverse) {
                    if (i < length - 1) {
                        i++;
                    } else {
                        i = 0;
                    }
                } else {
                    if (i > 0) {
                        i--;
                    } else {
                        i = length - 1;
                    }
                }
                break;
            default:
                break;
        }
        return i;
    }

    /**
     * moves the animator to the next frame index and returns the associated Sprite instance.
     * @return the next frame sprite instance
     * @see #getAnimatorValue(int, int)
     */
    public Sprite next() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            animator = getAnimatorValue(NEXT, animator);
            monitor.notifyOnMonitor();
            return getCurrentSprite();
        }
    }

    /**
     * moves the animator to the previous frame index and returns the associated Sprite instance.
     * @return previous frame sprite
     * @see #getAnimatorValue(int, int)
     */
    public Sprite previous() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            animator = getAnimatorValue(PREVIOUS, animator);
            monitor.notifyOnMonitor();
            return getCurrentSprite();
        }
    }

    /**
     * rewinds to the animator first index value.
     */
    public void rewind() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (reverse) {
                animator = length - 1;
            } else {
                animator = 0;
            }
            monitor.notifyOnMonitor();
        }
    }

    /**
     * Forwards to the animator end index value.
     */
    public void end() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            if (!reverse) {
                animator = length - 1;
            } else {
                animator = 0;
            }
            monitor.notifyOnMonitor();
        }
    }

    /** clears the animation and cache resources. resources can be recovered with loadResource().
     *  a new thread is registered in the buffer threads map.
     *@return Thread id of that command
     * @see #waitForActivity(int)
     * @see #loadResource()
     */
    public Object clearResource() {
        stop();
        currentSprite = null;
        spm.clear();
        spm.cleanup();
        return super.clearResource();
    }

    /** caches the resources of this animation. a new thread is registered in the buffer threads map.
     *@return Thread id of that command
     * @see #waitForActivity(int)
     * @see #clearResource()
     */
    public Object loadResource() {
        int hash;
        stop();
        for (int i = 0; i < length; i++) {
            loadFrame(i).clearResource();
        }
        currentSprite = getCurrentSprite();
        return super.loadResource();
    }

    /** cancels any running Thread of that animation, including the animator timer */
    public void cancel() {
        int pty = Thread.currentThread().getPriority();
        Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        timer = null;
        clearResource();
        Thread.currentThread().setPriority(pty);
    }

    /**
     * removes current frame sprite instance from mapping, it cannot be recovered.
     * @deprecated functional but usually unused
     */
    public void remove() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            frames.remove(animator);
            monitor.notifyOnMonitor();
        }
    }

    /** checks for next frame sprite instance availability
     * @return true or false*/
    public boolean hasNext() {
        final CoalescedThreadsMonitor monitor = imageSynch;
        synchronized (monitor.getMonitor(false)) {
            int next = (reverse) ? -1 : +1;
            monitor.notifyOnMonitor();
            return frames.containsKey(animator + next);
        }
    }

    /** overrides the paintComponent method to get the animation painted as a Swing component
     * @param g the graphics to paint on */
    protected void paintComponent(Graphics g) {
        try {
            draw(this, (Graphics2D) g);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * change cache capacity to allow more frames to be mapped
     * @param n the new cache capacity
     * @see SpritesCacheManager#setListCapacity(int)
     */
    public void setCacheCapacity(int n) {
        spm.setListCapacity(n);
    }

    /** returns the cache capacity
     * @return the cache capacity */
    public int getCacheCapacity() {
        return spm.getListCapacity();
    }

    /** not used */
    public void pack() {
        throw new UnsupportedOperationException("Animations don't need to pack. call super.pack");
    }

    /** returns the cache ReferenceQueue that repeatly polls to free resource of unused sprites instance.
     * @return the ReferenceQueue that polls for sprites instance to free resources
     * @see SpritesCacheManager#_cacheBack
     */
    public ReferenceQueue getCacheBack() {
        return spm._cacheBack;
    }

    /** dis/enables transform. if it is disabled no transform will be made.
     * @see #setFlipEnabled(boolean, int)
     * @see #setZoomEnabled(boolean, double)
     * @param transform dis/enable*/
    public void setTransformEnabled(boolean transform) {
        this.transform = transform;
    }

    /** plays the associated Sound in this Animation
     * @return true or false, whether the sound played or not
     * @see #setSfx(String, boolean, int)*/
    public boolean playSfx() {
        if (sfx instanceof Sound) {
            sfx.play();
            return true;
        } else {
            return false;
        }
    }

    /** sets this animation ThreadGroup to use as parent wrapper for all Threads
     * running in this animation instance.
     * @param tg ThreadGroup to use as parent, if null the current ThreadGroup that calls the method is used.
     * @see sf3.system#CoalescedThreadsMonitor(ThreadGroup, String)
     */
    public void setThreadGroup(ThreadGroup tg) {
        super.setThreadGroup(tg);       
    }
}
