/*-------------------------------------------------------------------------
 *
 * test_decoding.c
 *		  example logical decoding output plugin
 *
 * Copyright (c) 2012-2015, PostgreSQL Global Development Group
 *
 * IDENTIFICATION
 *		  contrib/test_decoding/test_decoding.c
 *
 *-------------------------------------------------------------------------
 */

#ifndef inc_pg_entee
#define inc_pg_entee
#include "pg_entete.h"
#endif
#include "decodeext.h"
#ifdef PG_MODULE_MAGIC
PG_MODULE_MAGIC;
#endif

/* These must be available to pg_dlsym() */
PGDLLEXPORT void _PG_init(void);
PGDLLEXPORT void _PG_output_plugin_init(OutputPluginCallbacks *cb);

typedef struct
{
	MemoryContext context;
	bool		include_xids;
	bool		include_timestamp;
	bool		skip_empty_xacts;
	bool		xact_wrote_changes;
	bool		only_local;
} TestDecodingData;

static void pg_decode_startup(LogicalDecodingContext *ctx, OutputPluginOptions *opt, bool is_init);
static void pg_decode_shutdown(LogicalDecodingContext *ctx);
static void pg_decode_begin_txn(LogicalDecodingContext *ctx, ReorderBufferTXN *txn);
static void pg_output_begin(LogicalDecodingContext *ctx, TestDecodingData *data, ReorderBufferTXN *txn, bool last_write);
static void pg_decode_commit_txn(LogicalDecodingContext *ctx, ReorderBufferTXN *txn, XLogRecPtr commit_lsn);
static void pg_decode_change(LogicalDecodingContext *ctx, ReorderBufferTXN *txn, Relation rel, ReorderBufferChange *change);
static bool pg_decode_filter(LogicalDecodingContext *ctx, RepOriginId origin_id);

void
_PG_init(void)
{
	/* other plugins can perform things here */
}

/* specify output plugin callbacks */
void
_PG_output_plugin_init(OutputPluginCallbacks *cb)
{
	AssertVariableIsOfType(&_PG_output_plugin_init, LogicalOutputPluginInit);

	cb->startup_cb = pg_decode_startup;
	cb->begin_cb = pg_decode_begin_txn;
	cb->change_cb = pg_decode_change;
	cb->commit_cb = pg_decode_commit_txn;
	cb->filter_by_origin_cb = pg_decode_filter;
	cb->shutdown_cb = pg_decode_shutdown;
}


/* initialize this plugin */
static void
pg_decode_startup(LogicalDecodingContext *ctx, OutputPluginOptions *opt, bool is_init)
{
	ListCell   *option;
	TestDecodingData *data;

	data = palloc0(sizeof(TestDecodingData));
	data->context = AllocSetContextCreate(ctx->context,
										  "text conversion context",
										  ALLOCSET_DEFAULT_MINSIZE,
										  ALLOCSET_DEFAULT_INITSIZE,
										  ALLOCSET_DEFAULT_MAXSIZE);
	data->include_xids = true;
	data->include_timestamp = false;
	data->skip_empty_xacts = false;
	data->only_local = false;

	ctx->output_plugin_private = data;

	opt->output_type = OUTPUT_PLUGIN_TEXTUAL_OUTPUT;

	foreach(option, ctx->output_plugin_options)
	{
		DefElem    *elem = lfirst(option);

		Assert(elem->arg == NULL || IsA(elem->arg, String));

		if (strcmp(elem->defname, "include-xids") == 0)
		{
			/* if option does not provide a value, it means its value is true */
			if (elem->arg == NULL)
				data->include_xids = true;
			else if (!parse_bool(strVal(elem->arg), &data->include_xids))
				ereport(ERROR,
						(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				  errmsg("could not parse value \"%s\" for parameter \"%s\"",
						 strVal(elem->arg), elem->defname)));
		}
		else if (strcmp(elem->defname, "include-timestamp") == 0)
		{
			if (elem->arg == NULL)
				data->include_timestamp = true;
			else if (!parse_bool(strVal(elem->arg), &data->include_timestamp))
				ereport(ERROR,
						(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				  errmsg("could not parse value \"%s\" for parameter \"%s\"",
						 strVal(elem->arg), elem->defname)));
		}
		else if (strcmp(elem->defname, "force-binary") == 0)
		{
			bool		force_binary;

			if (elem->arg == NULL)
				continue;
			else if (!parse_bool(strVal(elem->arg), &force_binary))
				ereport(ERROR,
						(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				  errmsg("could not parse value \"%s\" for parameter \"%s\"",
						 strVal(elem->arg), elem->defname)));

			if (force_binary)
				opt->output_type = OUTPUT_PLUGIN_BINARY_OUTPUT;
		}
		else if (strcmp(elem->defname, "skip-empty-xacts") == 0)
		{

			if (elem->arg == NULL)
				data->skip_empty_xacts = true;
			else if (!parse_bool(strVal(elem->arg), &data->skip_empty_xacts))
				ereport(ERROR,
						(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				  errmsg("could not parse value \"%s\" for parameter \"%s\"",
						 strVal(elem->arg), elem->defname)));
		}
		else if (strcmp(elem->defname, "only-local") == 0)
		{

			if (elem->arg == NULL)
				data->only_local = true;
			else if (!parse_bool(strVal(elem->arg), &data->only_local))
				ereport(ERROR,
						(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
				  errmsg("could not parse value \"%s\" for parameter \"%s\"",
						 strVal(elem->arg), elem->defname)));
		}
		else
		{
			ereport(ERROR,
					(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
					 errmsg("option \"%s\" = \"%s\" is unknown",
							elem->defname,
							elem->arg ? strVal(elem->arg) : "(null)")));
		}
	}
}

/* cleanup this plugin's resources */
static void
pg_decode_shutdown(LogicalDecodingContext *ctx)
{
	TestDecodingData *data = ctx->output_plugin_private;

	/* cleanup our own resources via memory context reset */
	MemoryContextDelete(data->context);
}

/* BEGIN callback */
static void
pg_decode_begin_txn(LogicalDecodingContext *ctx, ReorderBufferTXN *txn)
{
	TestDecodingData *data = ctx->output_plugin_private;

	data->xact_wrote_changes = false;
	if (data->skip_empty_xacts)
		return;

	pg_output_begin(ctx, data, txn, true);
}

static void
pg_output_begin(LogicalDecodingContext *ctx, TestDecodingData *data, ReorderBufferTXN *txn, bool last_write)
{
	OutputPluginPrepareWrite(ctx, last_write);
	/*if (data->include_xids)
		appendStringInfo(ctx->out, "BEGIN %u", txn->xid);
	else*/
		appendStringInfoString(ctx->out, "BEGIN");
	OutputPluginWrite(ctx, last_write);
}

/* COMMIT callback */
static void
pg_decode_commit_txn(LogicalDecodingContext *ctx, ReorderBufferTXN *txn,
					 XLogRecPtr commit_lsn)
{
	TestDecodingData *data = ctx->output_plugin_private;

	if (data->skip_empty_xacts && !data->xact_wrote_changes)
		return;

	OutputPluginPrepareWrite(ctx, true);
	/*if (data->include_xids)
		appendStringInfo(ctx->out, "COMMIT %u", txn->xid);
	else*/
		appendStringInfoString(ctx->out, "COMMIT");

	if (data->include_timestamp)
		appendStringInfo(ctx->out, " (at %s)",
						 timestamptz_to_str(txn->commit_time));

	OutputPluginWrite(ctx, true);
}

static bool
pg_decode_filter(LogicalDecodingContext *ctx,
				 RepOriginId origin_id)
{
	TestDecodingData *data = ctx->output_plugin_private;

	if (data->only_local && origin_id != InvalidRepOriginId)
		return true;
	return false;
}

/*
 * Print literal `outputstr' already represented as string of type `typid'
 * into stringbuf `s'.
 *
 * Some builtin types aren't quoted, the rest is quoted. Escaping is done as
 * if standard_conforming_strings were enabled.
 */
static void
print_literal(StringInfo s, Oid typid, char *outputstr)
{
	const char *valptr;

	switch (typid)
	{
		case INT2OID:
		case INT4OID:
		case INT8OID:
		case OIDOID:
		case FLOAT4OID:
		case FLOAT8OID:
		case NUMERICOID:
			/* NB: We don't care about Inf, NaN et al. */
			appendStringInfoString(s, outputstr);
			break;

		case BITOID:
		case VARBITOID:
			appendStringInfo(s, "B'%s'", outputstr);
			break;

		case BOOLOID:
			if (strcmp(outputstr, "t") == 0)
				appendStringInfoString(s, "true");
			else
				appendStringInfoString(s, "false");
			break;

		default:
			appendStringInfoChar(s, '\'');
			for (valptr = outputstr; *valptr; valptr++)
			{
				char		ch = *valptr;

				if (SQL_STR_DOUBLE(ch, false))
					appendStringInfoChar(s, ch);
				appendStringInfoChar(s, ch);
			}
			appendStringInfoChar(s, '\'');
			break;
	}
}

/* print the tuple 'tuple' into the StringInfo s */

static void
tuple_to_stringinfo(StringInfo s, TupleDesc tupdesc, HeapTuple tuple, bool skip_nulls, int operation, char *tb)
{
	/*
	operation:
	1=insert 
	2=update sans oldtuple
	3=update avec oldtuple
	4=delete
	*/
	int			natt;
	bool attispk=false;
	StringInfo lescolonnes=makeStringInfo();	
	StringInfo lesvaleurs=makeStringInfo();
	StringInfo lescolonnespk=makeStringInfo();	
	StringInfo lesvaleurspk=makeStringInfo();
	

	/* colonne OID non traitee*/
	
	appendStringInfoChar(lescolonnes, '(');
	appendStringInfoChar(lesvaleurs, '(');
	appendStringInfoChar(lescolonnespk, '(');
	appendStringInfoChar(lesvaleurspk, '(');
	/* print all columns individually */
	for (natt = 0; natt < tupdesc->natts; natt++)
	{
		Form_pg_attribute attr; /* the attribute itself */
		Oid			typid;		/* type of current attribute */
		Oid			typoutput;	/* output function */
		bool		typisvarlena;
		Datum		origval;	/* possibly toasted Datum */
		bool		isnull;		/* column is null? */

		attr = tupdesc->attrs[natt];
		attispk=att_est_pk(NameStr(attr->attname), tb);

		/* don't print dropped columns, we can't be sure everything is  available for them */
		if (attr->attisdropped) {continue;}

		/* Don't print system columns, oid will already have been printed if  present.*/
		if (attr->attnum < 0){continue;}

		typid = attr->atttypid;

		/* get Datum from tuple */
		origval = heap_getattr(tuple, natt + 1, tupdesc, &isnull);

		if (isnull && skip_nulls){continue;}

		/* print attribute name */
		if(natt!=0)
		{
			appendStringInfoString(lescolonnes, ", ");
			if(attispk==true)
			{appendStringInfoString(lescolonnespk, ", ");}
		}
		appendStringInfoString(lescolonnes, quote_identifier(NameStr(attr->attname)));
		if(attispk==true)
		{appendStringInfoString(lescolonnespk, quote_identifier(NameStr(attr->attname)));}
		
				
		/* query output function */
		getTypeOutputInfo(typid, &typoutput, &typisvarlena);

		/* print data */
		if(natt!=0)
		{
			appendStringInfoString(lesvaleurs, ", ");
			if(attispk==true)
			{appendStringInfoString(lesvaleurspk, ", ");}
		}
		if (isnull)
			appendStringInfoString(s, "null");		
		else if (!typisvarlena)
			{
				print_literal(lesvaleurs, typid, OidOutputFunctionCall(typoutput, origval));
				if(attispk==true)
				{print_literal(lesvaleurspk, typid, OidOutputFunctionCall(typoutput, origval));}

			}
		else
		{
			Datum		val;	/* definitely detoasted Datum */

			val = PointerGetDatum(PG_DETOAST_DATUM(origval));
			print_literal(lesvaleurs, typid, OidOutputFunctionCall(typoutput, val));
			if(attispk==true)
			{print_literal(lesvaleurspk, typid, OidOutputFunctionCall(typoutput, val));}
		}
	}
		appendStringInfoChar(lescolonnes, ')');
		appendStringInfoChar(lesvaleurs, ')');
		appendStringInfoChar(lescolonnespk, ')');
		appendStringInfoChar(lesvaleurspk, ')');

		
		if(operation==1)//INSERT INTO T(...) VALUES(...)
		{
			appendStringInfoString(s,lescolonnes->data);
			appendStringInfoString(s," VALUES");
			appendStringInfoString(s,lesvaleurs->data);
		}
		
		if(operation==2)
		{
			appendStringInfoString(s,lescolonnes->data);
			appendStringInfoChar(s,'=');
			appendStringInfoString(s,lesvaleurs->data);			
		}
		

		if(operation==3)//UPDATE T SET (...)=(...) WHERE (...)=(...)
		{			
			appendStringInfoString(s,lescolonnes->data);
			appendStringInfoChar(s,'=');
			appendStringInfoString(s,lesvaleurs->data);
			appendStringInfoString(s," WHERE ");
			appendStringInfoString(s,lescolonnespk->data);
			appendStringInfoChar(s,'=');
			appendStringInfoString(s,lesvaleurspk->data);
		}
		
		if(operation==4)//DELETE FROM ... WHERE (...)=(...)
		{
			appendStringInfoString(s," WHERE ");
			appendStringInfoString(s,lescolonnes->data);
			appendStringInfoChar(s,'=');
			appendStringInfoString(s,lesvaleurs->data);
		}
		
		pfree(lescolonnes);
		pfree(lesvaleurs);
		pfree(lescolonnespk);
		pfree(lesvaleurspk);
}
/*
 * callback for individual changed tuples
 */
static void
pg_decode_change(LogicalDecodingContext *ctx, ReorderBufferTXN *txn, Relation relation, ReorderBufferChange *change)
{
	TestDecodingData *data;
	Form_pg_class class_form;
	TupleDesc	tupdesc;
	MemoryContext old;

	data = ctx->output_plugin_private;

	/* output BEGIN if we haven't yet */
	if (data->skip_empty_xacts && !data->xact_wrote_changes)
	{
		pg_output_begin(ctx, data, txn, false);
	}
	data->xact_wrote_changes = true;

	class_form = RelationGetForm(relation);
	tupdesc = RelationGetDescr(relation);
	
	/* Avoid leaking memory by using and resetting our own context */
	old = MemoryContextSwitchTo(data->context);

	OutputPluginPrepareWrite(ctx, true);
	

	switch (change->action)
	{
		case REORDER_BUFFER_CHANGE_INSERT:
			appendStringInfoString(ctx->out, " INSERT INTO ");
			break;																							
		case REORDER_BUFFER_CHANGE_UPDATE:
			appendStringInfoString(ctx->out, " UPDATE ");
			break;
		case REORDER_BUFFER_CHANGE_DELETE:
			appendStringInfoString(ctx->out, " DELETE FROM ");	
		break;
		default:
			Assert(false);
	}
	
	appendStringInfoString(ctx->out, 
						quote_qualified_identifier(
									get_namespace_name(get_rel_namespace(RelationGetRelid(relation))),
									NameStr(class_form->relname)
												)
						);
	if(change->action==REORDER_BUFFER_CHANGE_UPDATE){appendStringInfoString(ctx->out," SET ");}
	
	
	

	switch (change->action)
	{
		case REORDER_BUFFER_CHANGE_INSERT:
			
			/*if (change->data.tp.newtuple == NULL)
				appendStringInfoString(ctx->out, " (no-tuple-data)");
			else*/
				tuple_to_stringinfo(ctx->out, tupdesc, &change->data.tp.newtuple->tuple, false, 1, NameStr(class_form->relname));
			break;
			
		case REORDER_BUFFER_CHANGE_UPDATE:			
			if (change->data.tp.newtuple != NULL && change->data.tp.oldtuple == NULL)
			{
				tuple_to_stringinfo(ctx->out, tupdesc, &change->data.tp.newtuple->tuple, false, 3, NameStr(class_form->relname));
			}
			else if (change->data.tp.oldtuple != NULL && change->data.tp.newtuple != NULL)
			{
				tuple_to_stringinfo(ctx->out, tupdesc, &change->data.tp.newtuple->tuple, false,2, NameStr(class_form->relname));
				appendStringInfoString(ctx->out, " WHERE ");
				tuple_to_stringinfo(ctx->out, tupdesc, &change->data.tp.oldtuple->tuple, true,2, NameStr(class_form->relname));
			}
			else
			{Assert(false);}
			break;
			
		case REORDER_BUFFER_CHANGE_DELETE:						
			if (change->data.tp.oldtuple != NULL)
			{
				tuple_to_stringinfo(ctx->out, tupdesc, &change->data.tp.oldtuple->tuple, true, 4, NameStr(class_form->relname));
			}else{Assert(false);}
			break;
		default:
			Assert(false);
	}

	MemoryContextSwitchTo(old);
	MemoryContextReset(data->context);

	OutputPluginWrite(ctx, true);
}
