package com.developpez.adiguba.annotation;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.annotation.processing.Filer;
import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.util.Elements;
import javax.lang.model.util.SimpleElementVisitor6;
import javax.tools.FileObject;
import javax.tools.StandardLocation;
import javax.tools.Diagnostic.Kind;

/**
 * Classe permettant d'analyser les mthodes/constructeurs d'une classe/interface.
 * @author adiGuba
 */
class ParameterNamesVisitor extends SimpleElementVisitor6<Void, Void> {
	
	final Element root;
	final Properties methods = new Properties();
	
	public ParameterNamesVisitor(Element root) {
		this.root = (Element) root;
	}
	
	/**
	 * Mthode appele automatiquement pour toutes les mthodes/constructeurs :
	 */
	@Override
	public Void visitExecutable(ExecutableElement e, Void p) {
		// On rcupre les paramtres de la mthodes/constructeur :
		List<? extends VariableElement> parameters = e.getParameters();
		// On gnre la clef (nom de la mthode + type des paramtres)
		StringBuilder key = new StringBuilder(e.getSimpleName()).append("(");
		// On gnre la valeur (nom des paramtres)
		StringBuilder value = new StringBuilder();
		
		Iterator<? extends VariableElement> iter = parameters.iterator();
		if (iter.hasNext()) {
			VariableElement parameter = iter.next();
			key.append(parameter.asType());
			value.append(parameter.getSimpleName());
			while (iter.hasNext()) {
				parameter = iter.next();
				key.append(",").append(parameter.asType());
				value.append(",").append(parameter.getSimpleName());
			}
		}
		key.append(")");
		
		// Et on ajoute le couple clef/valeur dans un fichier *.properties :
		this.methods.setProperty(key.toString(), value.toString());			
		return null;
	}
	
	/**
	 * Mthode permettant de gnrer le fichier contenant le nom des mthodes :
	 * @param processingEnv
	 */	
	public void store(ProcessingEnvironment processingEnv) {
		Elements elements = processingEnv.getElementUtils();
		Filer filer = processingEnv.getFiler();
		
		try {
			// On cre le fichier :
			FileObject file = filer.createResource(
					StandardLocation.CLASS_OUTPUT,
					elements.getPackageOf(this.root).getQualifiedName(),
					this.root.getSimpleName() + ".methods",
					this.root);
			// Et on enregistre le fichier properties 
			OutputStream out = file.openOutputStream();
			try {
				this.methods.store(out, "Methods for " + this.root);
			} finally {
				out.close();
			}
		} catch (IOException e) {
			// En cas d'erreur :
			processingEnv.getMessager().printMessage(
				Kind.ERROR, e.getMessage(), this.root);
		}
	}
}